// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'app_state.dart';
import 'constants.dart';
import 'copy_move_confirm_button.dart';
import 'db_manager.dart';
import 'location.dart';
import 'destination_chooser_header.dart';
import 'destination_list_radio_group.dart';
import 'enum.dart';
import 'screen.dart';

/// Show a dialog to choose a destination for copied/moved items.
class DestinationChooser extends StatefulWidget {
  final Location locationFrom;
  final bool isMoveMode;
  final Function chosenLocationSetter;
  final Function chosenLocationGetter;
  final Function moveModeSetter;
  final Function copyFunction;
  final Location? chosenLocation;
  final int? itemCount;
  final CopyOperationType copyOperationType;

  const DestinationChooser({
    super.key,
    required this.locationFrom,
    required this.isMoveMode,
    required this.moveModeSetter,
    required this.chosenLocation,
    required this.chosenLocationSetter,
    required this.chosenLocationGetter,
    required this.copyFunction,
    this.itemCount,
    this.copyOperationType = CopyOperationType.itemsFromTable,
  });

  @override
  State<DestinationChooser> createState() => _DestinationChooserState();
}

class _DestinationChooserState extends State<DestinationChooser> {
  final ScrollController _scrollController = ScrollController();

  @override
  Widget build(BuildContext context) {
    final Map dialogLayout = Screen.isCompact(context)
        ? BS.dialogLayoutCompact
        : BS.dialogLayoutStandard;

    final List<String> validForCopyDbPaths = _getValidDestinationDbPaths();

    final List<Map> simpleDestinationInfo = _getSimpleDestinationInfo(validForCopyDbPaths);

    final List<Widget> destinations = [
      ...simpleDestinationInfo.map((Map dbAndPlaylists) {
        ///
        /// Loop through Databases
        return Container(
          constraints: BoxConstraints.loose(
            Size(
              BS.destinationListWidth,
              widget.copyOperationType == CopyOperationType.itemsFromTable
                  ? BS.destinationListHeight
                  : BS.destinationSingleItemHeight,
            ),
          ),
          child: DestinationListRadioGroup(
            locationFrom: widget.locationFrom,
            locationTo: Location(
              dbPath: dbAndPlaylists['dbPath'],
              tableType:
                  (widget.locationFrom.tableType == TableType.remotePlaylists ||
                      widget.locationFrom.tableType == TableType.subscriptions)
                  ? widget.locationFrom.tableType
                  : null,
              tableId:
                  (widget.locationFrom.tableType == TableType.remotePlaylists ||
                      widget.locationFrom.tableType == TableType.subscriptions)
                  ? widget.locationFrom.tableId
                  : null,
            ),

            /// Each [DestinationListRadioButton] creates a [Location] describing its location.
            chosenLocationGetter: widget.chosenLocationGetter,
            chosenLocationSetter: widget.chosenLocationSetter,
            tables: switch (widget.copyOperationType) {
              CopyOperationType.fullTable => [],
              CopyOperationType.itemsFromTable => dbAndPlaylists['tables'],
            },
          ),
        );
      }),
    ];

    // top: dialogLayout['topPad'],
    // DestinationChooserHeader(
    // bottom: dialogLayout['bottomPad'],
    // Move mode checkbox and confirm button
    /// TODO Find better way of calculating this
    double maxHeight =
        MediaQuery.sizeOf(context).height -
        (BS.paddingLarge +
            BS.paddingLarge +
            BS.paddingLarge +
            BS.paddingLarge +
            BS.paddingLarge +
            BS.paddingLarge +
            BS.paddingLarge +
            dialogLayout['topPad'] +
            BS.fontSizeHeading +
            dialogLayout['bottomPad'] +
            BS.moveModeTextMaxHeight);

    // Main outer (dialog) box
    return Padding(
      padding: EdgeInsets.only(
        right: dialogLayout['rightPad'],
        bottom: dialogLayout['bottomPad'],
        left: dialogLayout['leftPad'],
      ),
      child: Column(
        mainAxisSize: MainAxisSize.min,
        crossAxisAlignment: CrossAxisAlignment.end,
        children: [
          // Title
          DestinationChooserHeader(
            validForCopyDbPaths: validForCopyDbPaths,
            tableTypeFrom: widget.locationFrom.tableType,
            itemCount: widget.itemCount,
            isMoveMode: widget.isMoveMode,
          ),

          // List of Databases
          ConstrainedBox(
            constraints: BoxConstraints(maxHeight: maxHeight, minWidth: double.infinity),
            child: Padding(
              padding: EdgeInsets.only(
                left: dialogLayout['innerLeftPad'],
                right: dialogLayout['innerRightPad'],
              ),
              child: Scrollbar(
                controller: _scrollController,
                thumbVisibility: true,
                child: SingleChildScrollView(
                  padding: EdgeInsets.only(right: BS.paddingMedium),
                  controller: _scrollController,
                  child: Wrap(
                    alignment: WrapAlignment.start,
                    spacing: BS.paddingMedium,
                    runSpacing: BS.paddingMedium,
                    children: destinations,
                  ),
                ),
              ),
            ),
          ),

          // Move mode checkbox and confirm button
          //if (validForCopyDbPaths.isNotEmpty && widget.chosenLocation != null)
          Padding(
            padding: EdgeInsets.only(
              top: dialogLayout['innerTopPad'],
              left: dialogLayout['innerLeftPad'],
              right: dialogLayout['innerRightPad'],
            ),
            child: CopyMoveConfirmButton(
              buttonLabel: _getActionDescription(
                context: context,
                destination: widget.chosenLocation,
                isMoveMode: widget.isMoveMode,
              ),
              isMoveMode: widget.isMoveMode,
              moveModeDisabled: (widget.locationFrom.tableType == TableType.search),
              onMoveModeChanged: (bool? value) {
                widget.moveModeSetter(value);
              },
              onConfirmPressedCallback: (widget.chosenLocation == null) ? null : _executeCopyOrMove,
            ),
          ),
        ],
      ),
    );
  }

  /// Get a list of paths that we can copy to, omitting the current table.
  List<String> _getValidDestinationDbPaths() {
    /// If this is a 'selected items' copy, we need to show the current
    /// database as well as the other opened databases, as items can be copied
    /// from one playlist to another, but otherwise (meaning it's an entire table/playlist being
    /// copied) we need to remove the current database.
    /// Search tables are a special case as they can be copied/duplicated to create a new playlist.
    List<String> paths = DbManager.databases.keys;
    if ((widget.copyOperationType == CopyOperationType.fullTable &&
            widget.locationFrom.tableType != TableType.search) ||
        (widget.copyOperationType == CopyOperationType.itemsFromTable &&
            (widget.locationFrom.tableType == TableType.subscriptions ||
                widget.locationFrom.tableType == TableType.remotePlaylists))) {
      paths.removeWhere((dbPath) => dbPath == widget.locationFrom.dbPath);
    }

    return paths;
  }

  /// Create simplified list of databases/playlists, containing only the info we need to draw the
  /// radio buttons (each table's id and name).
  List<Map> _getSimpleDestinationInfo(List<String> dbPaths) {
    List<Map> info = dbPaths.map((dbPath) {
      return {
        'dbPath': dbPath,
        'tables': [
          /// Local playlists
          if (widget.locationFrom.tableType == TableType.localPlaylist ||
              widget.locationFrom.tableType == TableType.search)
            ...DbManager.getLocalPlaylists(dbPath).keys
                .map(
                  (tableId) =>
                      /// If skipping same playlist, return an empty Map which we can remove later.
                      (dbPath == widget.locationFrom.dbPath &&
                          tableId == widget.locationFrom.tableId)
                      ? {}
                      : {'id': tableId, 'name': DbManager.getLocalPlaylistName(dbPath, tableId)},
                )
                .where((item) => item.isNotEmpty),
        ],
      };
    }).toList();

    return info;
  }

  /// Get a description for the action in its current state, to be displayed
  /// with the [CopyMoveConfirmButton].
  String _getActionDescription({
    required BuildContext context,
    required Location? destination,
    required bool isMoveMode,
  }) {
    String description = '';
    String shortDescription = '';

    String itemDescription = widget.copyOperationType == CopyOperationType.fullTable
        ? 'playlist'
        : switch (widget.locationFrom.tableType) {
            TableType.search || TableType.localPlaylist => 'stream',
            TableType.remotePlaylists => 'bookmarked playlist',
            TableType.subscriptions => 'channel',
            _ => '[Undefined table type]',
          };

    /// All above can be pluralised very simply
    if (widget.itemCount != null && widget.itemCount != 1) {
      itemDescription += 's';
    }

    String pathDescription = '';

    //AppState.debug('destination.dbPath: ${destination.dbPath}');
    //AppState.debug('destination.tableId: ${destination.tableId}');

    if (destination != null) {
      if (widget.locationFrom.tableType == TableType.localPlaylist ||
          widget.locationFrom.tableType == TableType.search) {
        pathDescription = switch (widget.copyOperationType) {
          CopyOperationType.fullTable => DbManager.getPrettyName(destination.dbPath),
          CopyOperationType.itemsFromTable =>
            "'${DbManager.getLocalPlaylistName(destination.dbPath, destination.tableId)}' in ${DbManager.getPrettyName(destination.dbPath)}",
        };
      } else {
        // If it's not a custom playlist we know it must be one of the other known tables.
        pathDescription =
            "'${BS.knownTables[widget.locationFrom.tableId]?['displayName']}' in ${DbManager.getPrettyName(destination.dbPath)}";
      }
    }

    description = '${isMoveMode ? 'Move' : 'Copy'} $itemDescription to $pathDescription';
    shortDescription = isMoveMode ? 'Move' : 'Copy';

    return Screen.isCompact(context) ? shortDescription : description;
  }

  void _executeCopyOrMove() {
    //AppState.debug('DestinationChooser::_executeCopyOrMove');
    if (widget.chosenLocation != null) {
      Location locationTo = widget.chosenLocation!;

      widget.copyFunction(
        locationFrom: widget.locationFrom,
        locationTo: widget.copyOperationType == CopyOperationType.itemsFromTable
            ? Location(
                dbPath: locationTo.dbPath,
                tableType: locationTo.tableType,
                tableId: locationTo.tableId,
              )
            : Location(dbPath: locationTo.dbPath),
        isMoveMode: widget.isMoveMode,
      );

      if (context.mounted) {
        Navigator.of(context).pop();
      }
    }
  }
}
