// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'checkbox_labelled.dart';
import 'constants.dart';
import 'tooltip_icon_button.dart';

/// 'Confirm' button for copy/move operations.
///
/// - Will be shown with different labels depending on whether this is a 'copy' or a 'move' operation
/// - Contains a toggle for 'move'/'copy' modes, which when pressed with fire a callback `onMoveModeChanged()`
/// - When the main button is pressed (so the copy or move is being confirmed), `onConfirmPressedCallback()` is called
class CopyMoveConfirmButton extends StatefulWidget {
  final String buttonLabel;
  final bool isMoveMode;
  final bool moveModeDisabled;
  final Function(bool?) onMoveModeChanged;
  final Function()? onConfirmPressedCallback;

  const CopyMoveConfirmButton({
    super.key,
    required this.buttonLabel,
    required this.isMoveMode,
    required this.onMoveModeChanged,
    required this.onConfirmPressedCallback,
    this.moveModeDisabled = false,
  });

  @override
  State<CopyMoveConfirmButton> createState() => _CopyMoveConfirmButtonState();
}

class _CopyMoveConfirmButtonState extends State<CopyMoveConfirmButton> {
  /// This flag is set moentarily while an operation completes (so between the time the copy/move
  /// button is pressed and the dialog closes).
  bool _disabledWhileExecuting = false;

  @override
  Widget build(BuildContext context) {
    return Row(
      mainAxisSize: MainAxisSize.min,
      spacing: BS.paddingLarge,
      //mainAxisAlignment: MainAxisAlignment.end,
      children: [
        CheckboxLabelled(
          labelText: 'Move mode',
          value: widget.isMoveMode,
          onChanged:
              widget.moveModeDisabled ||
                  _disabledWhileExecuting ||
                  widget.onConfirmPressedCallback == null
              ? null
              : widget.onMoveModeChanged,
        ),
        TooltipIconButton(
          text: widget.buttonLabel,
          iconData: Icons.drive_file_move_rounded,
          layout: TooltipIconButtonLayout.iconOnRight,
          onPressedCallback: _disabledWhileExecuting || widget.onConfirmPressedCallback == null
              ? null
              : () {
                  setState(() {
                    _disabledWhileExecuting = true;
                    Future.delayed(Duration(milliseconds: 500), () {
                      widget.onConfirmPressedCallback!();
                    });
                  });
                },
        ),
      ],
    );
  }
}
