/*
 * Copyright (c) 2024 Proton AG
 * This file is part of Proton AG and Proton Pass.
 *
 * Proton Pass is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Pass is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Pass.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.pass.features.sharing.sharingsummary

import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.material.Icon
import androidx.compose.material.Surface
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.pluralStringResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.tooling.preview.PreviewParameter
import me.proton.core.presentation.R
import proton.android.pass.commonui.api.PassTheme
import proton.android.pass.commonui.api.Spacing
import proton.android.pass.commonui.api.ThemePreviewProvider
import proton.android.pass.composecomponents.impl.container.BoxedIcon
import proton.android.pass.composecomponents.impl.text.Text
import proton.android.pass.domain.GroupId
import proton.android.pass.features.sharing.common.GroupTargetUiState
import proton.android.pass.features.sharing.common.InviteTargetUiState
import proton.android.pass.features.sharing.common.UserTargetUiState
import proton.android.pass.features.sharing.extensions.toStringResource
import proton.android.pass.features.sharing.sharingpermissions.SharingType
import proton.android.pass.composecomponents.impl.R as CompR

@Composable
internal fun AddressRowSection(
    modifier: Modifier = Modifier,
    inviteTarget: InviteTargetUiState,
    isRenameAdminToManagerEnabled: Boolean,
    onGroupMembersClick: (groupId: GroupId) -> Unit
) {
    Row(
        modifier = modifier,
        verticalAlignment = Alignment.CenterVertically,
        horizontalArrangement = Arrangement.spacedBy(space = Spacing.medium)
    ) {
        BoxedIcon(
            backgroundColor = PassTheme.colors.interactionNormMinor1,
            size = 40,
            shape = PassTheme.shapes.squircleMediumShape
        ) {
            Icon(
                painter = painterResource(R.drawable.ic_proton_envelope),
                contentDescription = null,
                tint = PassTheme.colors.interactionNormMajor2
            )
        }

        Column(
            modifier = Modifier.weight(1f),
            verticalArrangement = Arrangement.spacedBy(space = Spacing.extraSmall)
        ) {
            when (inviteTarget) {
                is UserTargetUiState -> {
                    Text.Body1Regular(
                        text = inviteTarget.displayName
                    )
                }

                is GroupTargetUiState -> {
                    if (inviteTarget.memberCount > 0) {
                        Row(
                            verticalAlignment = Alignment.CenterVertically,
                            horizontalArrangement = Arrangement.spacedBy(Spacing.extraSmall)
                        ) {
                            Text.Body1Regular(
                                text = inviteTarget.displayName
                            )
                            val label = pluralStringResource(
                                CompR.plurals.members_count,
                                inviteTarget.memberCount,
                                inviteTarget.memberCount
                            )
                            Text.Body1Regular(
                                text = "($label)",
                                color = PassTheme.colors.interactionNormMajor2,
                                modifier = Modifier.clickable {
                                    onGroupMembersClick(inviteTarget.groupId)
                                }
                            )
                        }
                    } else {
                        Text.Body1Regular(
                            text = inviteTarget.displayName
                        )
                    }
                }
            }

            Text.Body2Regular(
                text = stringResource(
                    inviteTarget.permission.toStringResource(
                        isRenameAdminToManagerEnabled
                    )
                ),
                color = PassTheme.colors.textWeak
            )
        }
    }
}


@[Preview Composable]
internal fun AddressRowSectionPreview(@PreviewParameter(ThemePreviewProvider::class) isDark: Boolean) {
    PassTheme(isDark = isDark) {
        Surface {
            AddressRowSection(
                inviteTarget = UserTargetUiState(
                    displayName = "my@test.email",
                    email = "my@test.email",
                    permission = SharingType.Write
                ),
                isRenameAdminToManagerEnabled = true,
                onGroupMembersClick = {}
            )
        }
    }
}
