/*
 * Copyright (c) 2023 Proton AG
 * This file is part of Proton AG and Proton Pass.
 *
 * Proton Pass is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Pass is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Pass.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.pass.data.impl.usecases.inappmessages

import app.cash.turbine.test
import com.google.common.truth.Truth.assertThat
import kotlinx.coroutines.test.runTest
import kotlinx.datetime.Clock
import me.proton.core.domain.entity.UserId
import org.junit.Before
import proton.android.pass.data.fakes.repositories.FakeInAppMessagesRepository
import proton.android.pass.data.fakes.usecases.FakeObserveCurrentUser
import proton.android.pass.domain.inappmessages.InAppMessageStatus
import proton.android.pass.preferences.LastTimeUserHasSeenIAMPreference
import proton.android.pass.preferences.FakeInternalSettingsRepository
import proton.android.pass.test.domain.InAppMessageTestFactory
import proton.android.pass.test.domain.UserTestFactory
import kotlin.test.Test
import kotlin.test.assertNull
import kotlin.time.Duration.Companion.minutes

internal class ObserveDeliverableBannerInAppMessagesImplTest {

    private lateinit var instance: ObserveDeliverableBannerInAppMessagesImpl
    private lateinit var observeCurrentUser: FakeObserveCurrentUser
    private lateinit var inAppMessagesRepository: FakeInAppMessagesRepository
    private lateinit var internalSettingsRepository: FakeInternalSettingsRepository
    private lateinit var clock: Clock
    private lateinit var userId: UserId

    @Before
    fun setup() {
        userId = UserId("test-user")
        observeCurrentUser = FakeObserveCurrentUser().apply {
            sendUser(UserTestFactory.create())
        }
        inAppMessagesRepository = FakeInAppMessagesRepository()
        internalSettingsRepository = FakeInternalSettingsRepository()
        clock = Clock.System
        instance = ObserveDeliverableBannerInAppMessagesImpl(
            observeCurrentUser = observeCurrentUser,
            inAppMessagesRepository = inAppMessagesRepository,
            internalSettingsRepository = internalSettingsRepository,
            clock = clock
        )
    }


    @Test
    fun `test no unread messages returns empty list`() = runTest {
        instance(userId).test {
            assertNull(awaitItem())
            cancelAndIgnoreRemainingEvents()
        }
    }

    @Test
    fun `test messages are not returned if last time is less than 30 minutes`() = runTest {
        val now = clock.now()
        internalSettingsRepository.setLastTimeUserHasSeenIAM(
            LastTimeUserHasSeenIAMPreference(userId, now.minus(29.minutes).epochSeconds)
        )
        val message = InAppMessageTestFactory.createBanner(
            state = InAppMessageStatus.Unread,
            range = InAppMessageTestFactory.createInAppMessageRange()
        )
        inAppMessagesRepository.addMessage(userId, message)
        instance(userId).test {
            assertNull(awaitItem())
            cancelAndIgnoreRemainingEvents()
        }
    }

    @Test
    fun `test messages are returned if last time is more than 30 minutes`() = runTest {
        val now = clock.now()
        internalSettingsRepository.setLastTimeUserHasSeenIAM(
            LastTimeUserHasSeenIAMPreference(userId, now.minus(31.minutes).epochSeconds)
        )
        val message = InAppMessageTestFactory.createBanner(
            state = InAppMessageStatus.Unread,
            range = InAppMessageTestFactory.createInAppMessageRange()
        )
        inAppMessagesRepository.addMessage(userId, message)
        instance(userId).test {
            val item = awaitItem()
            assertThat(item).isEqualTo(message)
            cancelAndIgnoreRemainingEvents()
        }
    }
}
