/*
 * Copyright (c) 2023 Proton AG
 * This file is part of Proton AG and Proton Pass.
 *
 * Proton Pass is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Pass is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Pass.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.pass.data.impl.usecases

import com.google.common.truth.Truth.assertThat
import kotlinx.coroutines.test.runTest
import me.proton.core.domain.entity.UserId
import org.junit.Before
import org.junit.Test
import proton.android.pass.data.api.usecases.RefreshSharesResult
import proton.android.pass.data.fakes.repositories.FakeItemRepository
import proton.android.pass.data.fakes.usecases.FakePromoteNewInviteToInvite
import proton.android.pass.data.fakes.usecases.FakeRefreshBreaches
import proton.android.pass.data.fakes.usecases.FakeRefreshGroupInvites
import proton.android.pass.data.fakes.usecases.FakeRefreshOrganizationSettings
import proton.android.pass.data.fakes.usecases.FakeRefreshSharesAndEnqueueSync
import proton.android.pass.data.fakes.usecases.FakeRefreshUserInvites
import proton.android.pass.data.fakes.usecases.FakeRefreshUserAccess
import proton.android.pass.data.fakes.usecases.simplelogin.FakeSyncSimpleLoginPendingAliases
import proton.android.pass.data.fakes.work.FakeWorkManagerFacade
import proton.android.pass.data.impl.fakes.FakeUserEventRepository
import proton.android.pass.data.impl.fakes.FakeShareRepository
import proton.android.pass.data.impl.work.FetchItemsWorker
import proton.android.pass.domain.ItemId
import proton.android.pass.domain.ShareId
import proton.android.pass.domain.UserEventId
import proton.android.pass.domain.events.EventToken
import proton.android.pass.domain.events.SyncEventInvitesChanged
import proton.android.pass.domain.events.SyncEventShare
import proton.android.pass.domain.events.SyncEventShareItem
import proton.android.pass.domain.events.UserEventList

internal class SyncUserEventsImplTest {

    private lateinit var instance: SyncUserEventsImpl
    private lateinit var userEventRepository: FakeUserEventRepository
    private lateinit var shareRepository: FakeShareRepository
    private lateinit var itemRepository: FakeItemRepository
    private lateinit var refreshSharesAndEnqueueSync: FakeRefreshSharesAndEnqueueSync
    private lateinit var workManagerFacade: FakeWorkManagerFacade
    private lateinit var refreshPlan: FakeRefreshUserAccess
    private lateinit var refreshUserInvites: FakeRefreshUserInvites
    private lateinit var refreshGroupInvites: FakeRefreshGroupInvites
    private lateinit var refreshBreaches: FakeRefreshBreaches
    private lateinit var syncPendingAliases: FakeSyncSimpleLoginPendingAliases
    private lateinit var promoteNewInviteToInvite: FakePromoteNewInviteToInvite
    private lateinit var refreshOrganizationSettings: FakeRefreshOrganizationSettings

    @Before
    fun setup() {
        userEventRepository = FakeUserEventRepository()
        shareRepository = FakeShareRepository()
        itemRepository = FakeItemRepository()
        refreshSharesAndEnqueueSync = FakeRefreshSharesAndEnqueueSync()
        workManagerFacade = FakeWorkManagerFacade()
        refreshPlan = FakeRefreshUserAccess()
        refreshUserInvites = FakeRefreshUserInvites()
        refreshGroupInvites = FakeRefreshGroupInvites()
        refreshBreaches = FakeRefreshBreaches()
        syncPendingAliases = FakeSyncSimpleLoginPendingAliases()
        promoteNewInviteToInvite = FakePromoteNewInviteToInvite()
        refreshOrganizationSettings = FakeRefreshOrganizationSettings()

        instance = SyncUserEventsImpl(
            userEventRepository = userEventRepository,
            shareRepository = shareRepository,
            itemRepository = itemRepository,
            refreshSharesAndEnqueueSync = refreshSharesAndEnqueueSync,
            workManagerFacade = workManagerFacade,
            refreshUserAccess = refreshPlan,
            refreshUserInvites = refreshUserInvites,
            refreshGroupInvites = refreshGroupInvites,
            syncPendingAliases = syncPendingAliases,
            promoteNewInviteToInvite = promoteNewInviteToInvite,
            refreshBreaches = refreshBreaches,
            refreshOrganizationSettings = refreshOrganizationSettings
        )
    }

    @Test
    fun `local events already up to date returns early`() = runTest {
        val localEventId = UserEventId(EVENT_ID_1)
        val remoteEventId = UserEventId(EVENT_ID_1)

        userEventRepository.setGetLatestEventIdFlow(localEventId)
        userEventRepository.setFetchLatestEventIdResult(remoteEventId)

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isEmpty()
    }

    @Test
    fun `no local event id triggers full refresh`() = runTest {
        val remoteEventId = UserEventId(EVENT_ID_1)

        userEventRepository.setGetLatestEventIdFlow(null)
        userEventRepository.setFetchLatestEventIdResult(remoteEventId)
        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = remoteEventId,
                fullRefresh = false
            )
        )
        refreshSharesAndEnqueueSync.setResult(RefreshSharesResult.NoSharesSkipped)

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    @Test
    fun `processes plan changed event`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                planChanged = true
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    @Test
    fun `processes shares created events`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        val share1 = SyncEventShare(ShareId(SHARE_ID_1), EventToken(TOKEN_1))
        val share2 = SyncEventShare(ShareId(SHARE_ID_2), EventToken(TOKEN_2))

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                sharesCreated = listOf(share1, share2)
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    @Test
    fun `processes shares updated events`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        val share1 = SyncEventShare(ShareId(SHARE_ID_1), EventToken(TOKEN_1))
        val share2 = SyncEventShare(ShareId(SHARE_ID_2), EventToken(TOKEN_2))

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                sharesUpdated = listOf(share1, share2)
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
        assertThat(shareRepository.refreshShareMemory()).hasSize(2)
    }

    @Test
    fun `processes shares deleted events`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        val share1 = SyncEventShare(ShareId(SHARE_ID_1), EventToken(TOKEN_1))
        val share2 = SyncEventShare(ShareId(SHARE_ID_2), EventToken(TOKEN_2))

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                sharesDeleted = listOf(share1, share2)
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    @Test
    fun `processes items updated events`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        val item1 = SyncEventShareItem(ShareId(SHARE_ID_1), ItemId(ITEM_ID_1), EventToken(TOKEN_1))
        val item2 = SyncEventShareItem(ShareId(SHARE_ID_2), ItemId(ITEM_ID_2), EventToken(TOKEN_2))

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                itemsUpdated = listOf(item1, item2)
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    @Test
    fun `processes items deleted events`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        val item1 = SyncEventShareItem(ShareId(SHARE_ID_1), ItemId(ITEM_ID_1), EventToken(TOKEN_1))
        val item2 = SyncEventShareItem(ShareId(SHARE_ID_1), ItemId(ITEM_ID_2), EventToken(TOKEN_2))
        val item3 = SyncEventShareItem(ShareId(SHARE_ID_2), ItemId(ITEM_ID_3), EventToken(TOKEN_3))

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                itemsDeleted = listOf(item1, item2, item3)
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    @Test
    fun `processes user invites changed event`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        val invitesChanged = SyncEventInvitesChanged(EventToken(TOKEN_1))

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                invitesChanged = invitesChanged
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    @Test
    fun `processes group invites changed event`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        val groupInvitesChanged = SyncEventInvitesChanged(EventToken(TOKEN_1))

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                groupInvitesChanged = groupInvitesChanged
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    @Test
    fun `processes pending alias changed event`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        val pendingAliasChanged = SyncEventInvitesChanged(EventToken(TOKEN_1))

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                pendingAliasToCreateChanged = pendingAliasChanged
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    @Test
    fun `processes new user invites changed events`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        val share1 = SyncEventShare(ShareId(SHARE_ID_1), EventToken(TOKEN_1))
        val share2 = SyncEventShare(ShareId(SHARE_ID_2), EventToken(TOKEN_2))

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                sharesWithInvitesToCreate = listOf(share1, share2)
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
        assertThat(promoteNewInviteToInvite.getInvocationMemory()).hasSize(2)
        assertThat(promoteNewInviteToInvite.getInvocationMemory()[0].second).isEqualTo(ShareId(SHARE_ID_1))
        assertThat(promoteNewInviteToInvite.getInvocationMemory()[1].second).isEqualTo(ShareId(SHARE_ID_2))
    }

    @Test
    fun `full refresh when event list has fullRefresh flag`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                fullRefresh = true
            )
        )
        refreshSharesAndEnqueueSync.setResult(RefreshSharesResult.NoSharesSkipped)

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    @Test
    fun `full refresh waits for worker when shares found and worker enqueued`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                fullRefresh = true
            )
        )
        refreshSharesAndEnqueueSync.setResult(
            RefreshSharesResult.SharesFound(
                shareIds = setOf(ShareId(SHARE_ID_1)),
                isWorkerEnqueued = true
            )
        )

        val uniqueName = FetchItemsWorker.getOneTimeUniqueWorkName(USER_ID)

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
        assertThat(workManagerFacade.getAwaitedWorkNames()).contains(uniqueName)
    }

    @Test
    fun `full refresh skips worker wait when no shares found`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                fullRefresh = true
            )
        )
        refreshSharesAndEnqueueSync.setResult(RefreshSharesResult.NoSharesSkipped)

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    @Test
    fun `processes multiple event batches when eventsPending is true`() = runTest {
        val eventId1 = UserEventId(EVENT_ID_1)
        val eventId2 = UserEventId(EVENT_ID_2)
        val eventId3 = UserEventId(EVENT_ID_3)
        setupBasicSync(eventId1)

        // First batch - events pending
        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId2,
                eventsPending = true
            )
        )

        // Second batch - events pending
        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId3,
                eventsPending = false
            )
        )

        instance.invoke(USER_ID)

        val storedEvents = userEventRepository.getStoreLatestEventIdMemory()
        assertThat(storedEvents).isNotEmpty()
    }

    @Test
    fun `stores latest event id after each batch`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                eventsPending = false
            )
        )

        instance.invoke(USER_ID)

        val storedEvents = userEventRepository.getStoreLatestEventIdMemory()
        assertThat(storedEvents).hasSize(1)
        assertThat(storedEvents[0].second).isEqualTo(eventId)
    }

    @Test
    fun `handles empty event lists`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                itemsUpdated = emptyList(),
                itemsDeleted = emptyList(),
                sharesCreated = emptyList(),
                sharesUpdated = emptyList(),
                sharesDeleted = emptyList()
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    @Test
    fun `handles null invites changed events`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                invitesChanged = null,
                groupInvitesChanged = null,
                pendingAliasToCreateChanged = null
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
        assertThat(workManagerFacade.getAwaitedWorkNames()).isEmpty()
    }

    @Test
    fun `handles work manager with finished state`() = runTest {
        val eventId = UserEventId(EVENT_ID_1)
        setupBasicSync(eventId)

        userEventRepository.setGetUserEventsResult(
            createUserEventList(
                lastEventId = eventId,
                fullRefresh = true
            )
        )
        refreshSharesAndEnqueueSync.setResult(
            RefreshSharesResult.SharesFound(
                shareIds = setOf(ShareId(SHARE_ID_1)),
                isWorkerEnqueued = true
            )
        )

        instance.invoke(USER_ID)

        assertThat(userEventRepository.getStoreLatestEventIdMemory()).isNotEmpty()
    }

    private fun setupBasicSync(initialEventId: UserEventId) {
        val localEventId = UserEventId("${initialEventId.id}-local")
        val remoteEventId = UserEventId("${initialEventId.id}-remote")
        userEventRepository.setGetLatestEventIdFlow(localEventId)
        userEventRepository.setFetchLatestEventIdResult(remoteEventId)
    }

    private fun createUserEventList(
        lastEventId: UserEventId,
        itemsUpdated: List<SyncEventShareItem> = emptyList(),
        itemsDeleted: List<SyncEventShareItem> = emptyList(),
        sharesCreated: List<SyncEventShare> = emptyList(),
        sharesUpdated: List<SyncEventShare> = emptyList(),
        sharesDeleted: List<SyncEventShare> = emptyList(),
        invitesChanged: SyncEventInvitesChanged? = null,
        groupInvitesChanged: SyncEventInvitesChanged? = null,
        pendingAliasToCreateChanged: SyncEventInvitesChanged? = null,
        breachUpdate: SyncEventInvitesChanged? = null,
        organizationInfoChanged: SyncEventInvitesChanged? = null,
        sharesWithInvitesToCreate: List<SyncEventShare> = emptyList(),
        planChanged: Boolean = false,
        eventsPending: Boolean = false,
        fullRefresh: Boolean = false
    ): UserEventList = UserEventList(
        lastEventId = lastEventId,
        itemsUpdated = itemsUpdated,
        itemsDeleted = itemsDeleted,
        aliasNoteChanged = emptyList(),
        sharesCreated = sharesCreated,
        sharesUpdated = sharesUpdated,
        sharesDeleted = sharesDeleted,
        foldersUpdated = emptyList(),
        foldersDeleted = emptyList(),
        invitesChanged = invitesChanged,
        groupInvitesChanged = groupInvitesChanged,
        pendingAliasToCreateChanged = pendingAliasToCreateChanged,
        sharesWithInvitesToCreate = sharesWithInvitesToCreate,
        breachUpdate = breachUpdate,
        organizationInfoChanged = organizationInfoChanged,
        refreshUser = planChanged,
        eventsPending = eventsPending,
        fullRefresh = fullRefresh
    )


    companion object {
        private val USER_ID = UserId("test-user-id")

        // Event IDs
        private const val EVENT_ID_1 = "event-1"
        private const val EVENT_ID_2 = "event-2"
        private const val EVENT_ID_3 = "event-3"

        // Share IDs
        private const val SHARE_ID_1 = "share-1"
        private const val SHARE_ID_2 = "share-2"

        // Item IDs
        private const val ITEM_ID_1 = "item-1"
        private const val ITEM_ID_2 = "item-2"
        private const val ITEM_ID_3 = "item-3"

        // Event Tokens
        private const val TOKEN_1 = "token-1"
        private const val TOKEN_2 = "token-2"
        private const val TOKEN_3 = "token-3"
    }

}

