/*
 * Copyright (c) 2023 Proton AG
 * This file is part of Proton AG and Proton Pass.
 *
 * Proton Pass is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Pass is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Pass.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.pass.data.impl.usecases

import kotlinx.coroutines.async
import kotlinx.coroutines.awaitAll
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.flow.first
import me.proton.core.domain.entity.UserId
import proton.android.pass.data.api.repositories.ItemRepository
import proton.android.pass.data.api.repositories.ShareRepository
import proton.android.pass.data.api.usecases.PromoteNewInviteToInvite
import proton.android.pass.data.api.usecases.RefreshBreaches
import proton.android.pass.data.api.usecases.RefreshGroupInvites
import proton.android.pass.data.api.usecases.RefreshUserAccess
import proton.android.pass.data.api.usecases.RefreshSharesAndEnqueueSync
import proton.android.pass.data.api.usecases.RefreshSharesResult
import proton.android.pass.data.api.usecases.RefreshUserInvites
import proton.android.pass.data.api.usecases.SyncUserEvents
import proton.android.pass.data.api.usecases.organization.RefreshOrganizationSettings
import proton.android.pass.data.api.usecases.simplelogin.SyncSimpleLoginPendingAliases
import proton.android.pass.data.api.work.WorkManagerFacade
import proton.android.pass.data.impl.repositories.UserEventRepository
import proton.android.pass.data.impl.work.FetchItemsWorker
import proton.android.pass.domain.UserEventId
import proton.android.pass.domain.events.SyncEventInvitesChanged
import proton.android.pass.domain.events.SyncEventShare
import proton.android.pass.domain.events.SyncEventShareItem
import proton.android.pass.domain.events.UserEventList
import proton.android.pass.log.api.PassLogger
import javax.inject.Inject

class SyncUserEventsImpl @Inject constructor(
    private val userEventRepository: UserEventRepository,
    private val shareRepository: ShareRepository,
    private val itemRepository: ItemRepository,
    private val refreshSharesAndEnqueueSync: RefreshSharesAndEnqueueSync,
    private val workManagerFacade: WorkManagerFacade,
    private val refreshUserAccess: RefreshUserAccess,
    private val refreshUserInvites: RefreshUserInvites,
    private val refreshGroupInvites: RefreshGroupInvites,
    private val syncPendingAliases: SyncSimpleLoginPendingAliases,
    private val promoteNewInviteToInvite: PromoteNewInviteToInvite,
    private val refreshBreaches: RefreshBreaches,
    private val refreshOrganizationSettings: RefreshOrganizationSettings
) : SyncUserEvents {

    override suspend fun invoke(userId: UserId, forceSync: Boolean) {
        PassLogger.d(TAG, "Syncing user events for $userId started (forceSync=$forceSync)")

        if (forceSync) {
            PassLogger.d(TAG, "Force sync requested, clearing local event ID")
            userEventRepository.deleteLatestEventId(userId)
        }

        val localEventId = getLocalEventId(userId)
        val remoteLatestEventId = userEventRepository.fetchLatestEventId(userId)

        if (localEventId == remoteLatestEventId && !forceSync) {
            PassLogger.d(TAG, "Local user events already up to date for $userId")
            return
        }

        processUserEvents(userId, localEventId ?: remoteLatestEventId)

        PassLogger.i(TAG, "Syncing user events for $userId finished")
    }

    private suspend fun getLocalEventId(userId: UserId): UserEventId? {
        val localEventId = userEventRepository.getLatestEventId(userId).first()
        if (localEventId == null) fullRefresh(userId)
        return localEventId
    }

    private suspend fun processUserEvents(userId: UserId, initialEventId: UserEventId) {
        var currentEventId = initialEventId

        do {
            val eventList = userEventRepository.getUserEvents(userId, currentEventId)
            if (eventList.fullRefresh) {
                fullRefresh(userId)
            } else {
                processIncrementalEvents(userId, eventList)
            }

            userEventRepository.storeLatestEventId(userId, eventList.lastEventId)
            PassLogger.i(TAG, "Fetched user events, eventsPending: ${eventList.eventsPending}")
            currentEventId = eventList.lastEventId
        } while (eventList.eventsPending)
    }

    private suspend fun processIncrementalEvents(userId: UserId, eventList: UserEventList) {
        PassLogger.i(TAG, "Processing events for $userId")

        if (eventList.refreshUser) {
            refreshUserAccess(userId)
        }

        processSharesCreated(userId, eventList.sharesCreated)
        processSharesUpdated(userId, eventList.sharesUpdated)
        processSharesDeleted(userId, eventList.sharesDeleted)
        processItemsUpdated(userId, eventList.itemsUpdated)
        processItemsDeleted(userId, eventList.itemsDeleted)
        processInvitesChanged(userId, eventList.invitesChanged)
        processGroupInvitesChanged(userId, eventList.groupInvitesChanged)
        processPendingAliasToCreateChanged(userId, eventList.pendingAliasToCreateChanged)
        processBreachUpdateChanged(userId, eventList.breachUpdate)
        processOrganizationUpdateChanged(userId, eventList.organizationInfoChanged)
        processNewUserInvitesChanged(userId, eventList.sharesWithInvitesToCreate)
    }

    private suspend fun processSharesCreated(userId: UserId, sharesCreated: List<SyncEventShare>) {
        sharesCreated.forEach { (shareId, token) ->
            shareRepository.recreateShare(userId, shareId, token)
        }
    }

    private suspend fun processSharesUpdated(userId: UserId, sharesUpdated: List<SyncEventShare>) {
        sharesUpdated.forEach { (shareId, token) ->
            shareRepository.refreshShare(userId, shareId, token)
        }
    }

    private suspend fun processItemsUpdated(userId: UserId, itemsUpdated: List<SyncEventShareItem>) {
        itemsUpdated.forEach { (shareId, itemId, token) ->
            itemRepository.refreshItem(userId, shareId, itemId, token)
        }
    }

    private suspend fun processSharesDeleted(userId: UserId, sharesDeleted: List<SyncEventShare>) {
        if (sharesDeleted.isNotEmpty()) {
            val sharesToDelete = sharesDeleted.map(SyncEventShare::shareId)
            shareRepository.deleteLocalShares(userId, sharesToDelete)
        }
    }

    private suspend fun processItemsDeleted(userId: UserId, itemsDeleted: List<SyncEventShareItem>) {
        if (itemsDeleted.isNotEmpty()) {
            val itemsToDelete = itemsDeleted
                .groupBy { it.shareId }
                .mapValues { values -> values.value.map(SyncEventShareItem::itemId) }
            itemRepository.deleteLocalItems(userId, itemsToDelete)
        }
    }

    private suspend fun processInvitesChanged(userId: UserId, invitesChanged: SyncEventInvitesChanged?) {
        invitesChanged?.let { refreshUserInvites(userId, it.eventToken) }
    }

    private suspend fun processGroupInvitesChanged(userId: UserId, invitesChanged: SyncEventInvitesChanged?) {
        invitesChanged?.let { refreshGroupInvites(userId, it.eventToken) }
    }

    private suspend fun processPendingAliasToCreateChanged(userId: UserId, changed: SyncEventInvitesChanged?) {
        changed?.let { syncPendingAliases(userId, false) }
    }

    private suspend fun processBreachUpdateChanged(userId: UserId, changed: SyncEventInvitesChanged?) {
        changed?.let { refreshBreaches(userId) }
    }

    private suspend fun processOrganizationUpdateChanged(userId: UserId, changed: SyncEventInvitesChanged?) {
        changed?.let { refreshOrganizationSettings(userId) }
    }

    private suspend fun processNewUserInvitesChanged(userId: UserId, sharesWithInvitesToCreate: List<SyncEventShare>) {
        if (sharesWithInvitesToCreate.isNotEmpty()) {
            sharesWithInvitesToCreate.forEach { (shareId, _) ->
                promoteNewInviteToInvite(userId, shareId)
            }
        }
    }

    private suspend fun fullRefresh(userId: UserId) = coroutineScope {
        PassLogger.i(TAG, "start full refresh")

        refreshUserAccess(userId)

        val refreshShares = refreshSharesAndEnqueueSync(
            userId = userId,
            syncType = RefreshSharesAndEnqueueSync.SyncType.FULL
        )

        if (refreshShares is RefreshSharesResult.SharesFound && refreshShares.isWorkerEnqueued) {
            PassLogger.i(TAG, "waiting worker")
            waitForFetchItemsWorker(userId)
            PassLogger.i(TAG, "worker finished")
        }

        val userInvitesDeferred = async {
            val result = refreshUserInvites(userId)
            PassLogger.i(TAG, "finished refreshUserInvites")
            result
        }

        val groupInvitesDeferred = async {
            val result = refreshGroupInvites(userId)
            PassLogger.i(TAG, "finished refreshGroupInvites")
            result
        }

        val syncPendingAliasesDeferred = async {
            val result = syncPendingAliases(userId, false)
            PassLogger.i(TAG, "finished syncPendingAliases")
            result
        }

        val refreshBreachesDeferred = async {
            val result = refreshBreaches(userId)
            PassLogger.i(TAG, "finished refreshBreaches")
            result
        }

        val refreshOrganizationSettingsDeferred = async {
            val result = refreshOrganizationSettings(userId)
            PassLogger.i(TAG, "finished refreshOrganizationSettings")
            result
        }

        awaitAll(
            userInvitesDeferred,
            groupInvitesDeferred,
            syncPendingAliasesDeferred,
            refreshBreachesDeferred,
            refreshOrganizationSettingsDeferred
        )

        PassLogger.i(TAG, "end full refresh")
    }

    private suspend fun waitForFetchItemsWorker(userId: UserId) {
        val uniqueName = FetchItemsWorker.getOneTimeUniqueWorkName(userId)
        workManagerFacade.awaitUniqueWorkFinished(uniqueName)
    }

    private companion object {
        private const val TAG = "SyncUserEventsImpl"
    }
}
