/*
 * Copyright (c) 2025 Proton AG
 * This file is part of Proton AG and Proton Authenticator.
 *
 * Proton Authenticator is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Authenticator is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Authenticator.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.authenticator.shared.ui.domain.modifiers

import android.graphics.BlendMode
import android.graphics.BlurMaskFilter
import android.os.Build
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.runtime.Stable
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.composed
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.drawBehind
import androidx.compose.ui.draw.drawWithContent
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.Brush
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.graphics.ImageShader
import androidx.compose.ui.graphics.Paint
import androidx.compose.ui.graphics.TileMode
import androidx.compose.ui.graphics.asAndroidBitmap
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.graphics.drawscope.drawIntoCanvas
import androidx.compose.ui.graphics.nativeCanvas
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.res.imageResource
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import androidx.core.content.ContextCompat
import androidx.core.graphics.createBitmap
import androidx.core.graphics.scale
import proton.android.authenticator.shared.ui.R
import proton.android.authenticator.shared.ui.domain.theme.Theme
import proton.android.authenticator.shared.ui.domain.theme.ThemeRadius
import proton.android.authenticator.shared.ui.domain.theme.ThemeSpacing
import proton.android.authenticator.shared.ui.domain.theme.ThemeThickness
import android.graphics.Canvas as AndroidCanvas

@Stable
fun Modifier.backgroundBlur(backgroundColor: Color, blurRadius: Dp) = drawBehind {
    android.graphics.Paint()
        .apply {
            isAntiAlias = true
            color = backgroundColor.toArgb()

            maskFilter = BlurMaskFilter(
                blurRadius.value,
                BlurMaskFilter.Blur.INNER
            )
        }
        .also { nativePaint ->
            drawIntoCanvas { canvas ->
                canvas.save()

                android.graphics.Rect(
                    -ThemeSpacing.Medium.value.toInt(),
                    -ThemeSpacing.Medium.value.toInt(),
                    size.width.toInt(),
                    size.height.toInt()
                ).also { rect ->
                    canvas.nativeCanvas.drawRect(rect, nativePaint)
                }

                canvas.restore()
            }
        }
}

@Stable
fun Modifier.backgroundAppBar(isBlurred: Boolean = false) = composed {
    val backgroundColor = Theme.colorScheme.backgroundTopBar

    applyIf(
        condition = isBlurred,
        ifTrue = {
            backgroundBlur(
                backgroundColor = backgroundColor,
                blurRadius = 10.dp
            )
        }
    )
}

@Stable
fun Modifier.backgroundPrimaryButton(isEnable: Boolean = true, blur: Dp = 24.dp) = composed {
    val shape = remember { CircleShape }

    val colors = if (isEnable) {
        listOf(
            Theme.colorScheme.buttonGradientTop,
            Theme.colorScheme.buttonGradientBottom
        )
    } else {
        listOf(
            Theme.colorScheme.buttonGradientTop.copy(alpha = 0.38f),
            Theme.colorScheme.buttonGradientBottom.copy(alpha = 0.38f)
        )
    }

    dropShadow(
        shape = shape,
        color = Theme.colorScheme.purpleAlpha25,
        blur = blur
    )
        .clip(shape = shape)
        .background(brush = Brush.verticalGradient(colors = colors))
        .border(
            shape = shape,
            width = ThemeThickness.Small,
            color = Theme.colorScheme.whiteAlpha12
        )
        .innerShadow(
            shape = shape,
            color = Theme.colorScheme.whiteAlpha25
        )
}

@Stable
fun Modifier.backgroundSecondaryButton() = composed {
    val shape = remember { CircleShape }

    clip(shape = shape)
        .background(color = Color.Transparent)
        .border(
            shape = shape,
            width = ThemeThickness.Small,
            color = Theme.colorScheme.backgroundButtonBorderWeak
        )
}

@Stable
fun Modifier.backgroundScreenGradient() = composed {
    val isDarkMode = isSystemInDarkTheme()
    val image = ImageBitmap.imageResource(R.drawable.bg_texture)
    val paintAlpha = remember { 38 }

    background(
        brush = Brush.verticalGradient(
            colors = listOf(
                Theme.colorScheme.backgroundGradientTop,
                Theme.colorScheme.backgroundGradientBottom
            )
        )
    ).drawBehind {
        Paint().asFrameworkPaint()
            .apply {
                alpha = paintAlpha
                isAntiAlias = true
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    blendMode = if (isDarkMode) BlendMode.SCREEN else BlendMode.DIFFERENCE
                }
                shader = ImageShader(
                    image = image,
                    tileModeX = TileMode.Repeated,
                    tileModeY = TileMode.Repeated
                )
            }
            .also { paint ->
                drawIntoCanvas { canvas ->
                    canvas.nativeCanvas.drawPaint(paint)
                }

                paint.reset()
            }
    }
}

@Stable
fun Modifier.backgroundDropdownMenu() = composed {
    val shape = remember { RoundedCornerShape(size = ThemeRadius.MediumSmall) }

    dropShadow(
        shape = CircleShape,
        color = Theme.colorScheme.blackAlpha8,
        offsetX = 0.dp,
        offsetY = 2.dp,
        blur = 4.dp
    )
        .clip(shape = shape)
        .background(color = Theme.colorScheme.backgroundDropdown)
        .innerShadow(
            shape = shape,
            color = Theme.colorScheme.whiteAlpha20,
            offsetX = 0.dp,
            offsetY = 1.dp,
            blur = 0.dp
        )
}

@Stable
fun Modifier.backgroundActionButton() = composed {
    val shape = remember { CircleShape }

    dropShadow(
        shape = shape,
        color = Theme.colorScheme.blackAlpha10,
        offsetX = 0.dp,
        offsetY = 2.dp,
        blur = 4.dp
    )
        .clip(shape = shape)
        .border(
            shape = shape,
            width = ThemeThickness.Small,
            brush = Brush.verticalGradient(
                colors = listOf(
                    Theme.colorScheme.actionButtonBorderGradientTop,
                    Theme.colorScheme.actionButtonBorderGradientBottom
                )
            )
        )
        .background(
            brush = Brush.verticalGradient(
                colors = listOf(
                    Theme.colorScheme.actionButtonBackgroundGradientTop,
                    Theme.colorScheme.actionButtonBackgroundGradientBottom
                )
            )
        )
        .innerShadow(
            shape = shape,
            color = Theme.colorScheme.whiteAlpha20,
            offsetX = 0.dp,
            offsetY = 1.dp,
            blur = 1.dp
        )
}

@Stable
fun Modifier.backgroundSection(applyShadow: Boolean = false) = composed {
    val shape = remember { RoundedCornerShape(size = ThemeRadius.Medium) }
    val shadowColor = Theme.colorScheme.blackAlpha12

    applyIf(
        condition = applyShadow,
        ifTrue = {
            dropShadow(
                shape = shape,
                color = shadowColor,
                offsetX = 0.dp,
                offsetY = 2.dp,
                blur = 8.dp
            )
        }
    )
        .clip(shape = shape)
        .border(
            width = ThemeThickness.Small,
            color = Theme.colorScheme.menuListBorder,
            shape = shape
        )
        .background(color = Theme.colorScheme.menuListBackground)

}

@Stable
fun Modifier.backgroundOnboarding() = composed {
    val backgroundImage = ImageBitmap.imageResource(id = R.drawable.preview_authenticator)

    drawWithContent {
        drawContent()

        drawIntoCanvas { canvas ->
            canvas.apply {
                val scaledBackgroundImage = backgroundImage
                    .asAndroidBitmap()
                    .scale(
                        width = size.width.toInt(),
                        height = size.width.div(backgroundImage.width).times(backgroundImage.height).toInt(),
                        filter = true
                    )

                drawImage(
                    image = scaledBackgroundImage.asImageBitmap(),
                    topLeft = Offset(
                        x = size.width.div(2) - scaledBackgroundImage.width.div(2),
                        y = 0f
                    )
                )
            }
        }
    }
}

@[Stable Suppress("LongMethod")]
fun Modifier.backgroundFireball(size: Dp) = composed {
    val context = LocalContext.current
    val density = LocalDensity.current
    val originalImageBitmap = ImageBitmap.imageResource(R.drawable.bg_texture_hexagons)

    val scaledImageBitmap: ImageBitmap = remember(originalImageBitmap, density) {
        val targetWidthPx = with(density) { size.toPx().toInt() }
        val targetHeightPx = with(density) { size.toPx().toInt() }
        val androidBitmap = originalImageBitmap.asAndroidBitmap()
        val scaledAndroidBitmap = androidBitmap.scale(targetWidthPx, targetHeightPx)
        scaledAndroidBitmap.asImageBitmap()
    }

    val vectorImageBitmap: ImageBitmap = remember {
        val drawable = ContextCompat.getDrawable(context, R.drawable.bg_fireball)
        val widthPx = (size.value * context.resources.displayMetrics.density).toInt()
        val heightPx = (size.value * context.resources.displayMetrics.density).toInt()
        val bitmap = createBitmap(widthPx, heightPx)
        val canvas = AndroidCanvas(bitmap)
        drawable?.setBounds(0, 0, canvas.width, canvas.height)
        drawable?.draw(canvas)
        bitmap.asImageBitmap()
    }

    drawBehind {
        drawImage(
            image = vectorImageBitmap,
            topLeft = Offset(
                x = (this.size.width - vectorImageBitmap.width) / 2,
                y = (this.size.height - vectorImageBitmap.height) / 2
            )
        )

        drawImage(
            image = scaledImageBitmap,
            topLeft = Offset(
                x = (this.size.width - scaledImageBitmap.width) / 2,
                y = (this.size.height - scaledImageBitmap.height) / 2
            )
        )
    }
        .innerShadow(
            shape = CircleShape,
            color = Color(color = 0x44FFD000),
            offsetX = 0.dp,
            offsetY = (-24).dp,
            blur = 21.dp
        )
        .innerShadow(
            shape = CircleShape,
            color = Color(color = 0x44FFAD2A),
            offsetX = (-13.5).dp,
            offsetY = (-10.5).dp,
            blur = 24.dp
        )
        .dropShadow(
            shape = CircleShape,
            color = Color(color = 0x20FF0000),
            offsetX = 0.dp,
            offsetY = 0.dp,
            blur = 104.dp
        )
        .dropShadow(
            shape = CircleShape,
            color = Color(color = 0x24FFC300),
            offsetX = 0.dp,
            offsetY = 4.dp,
            blur = 24.dp
        )
}
