/*
 * Copyright (c) 2025 Proton AG
 * This file is part of Proton AG and Proton Authenticator.
 *
 * Proton Authenticator is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Authenticator is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Authenticator.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.authenticator.shared.ui.domain.components.tabs

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Tab
import androidx.compose.material3.TabRow
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import proton.android.authenticator.shared.ui.domain.modifiers.applyIf
import proton.android.authenticator.shared.ui.domain.modifiers.backgroundDropdownMenu
import proton.android.authenticator.shared.ui.domain.theme.Theme
import proton.android.authenticator.shared.ui.domain.theme.ThemeRadius
import proton.android.authenticator.shared.ui.domain.theme.ThemeSpacing

@Composable
fun FormTab(
    title: String,
    selectedTabIndex: Int,
    tabs: List<String>,
    onTabSelected: (Int) -> Unit,
    modifier: Modifier = Modifier
) {
    Column(
        modifier = modifier,
        verticalArrangement = Arrangement.spacedBy(space = ThemeSpacing.Small)
    ) {
        Text(
            modifier = Modifier.padding(start = ThemeSpacing.Medium),
            text = title.uppercase(),
            color = Theme.colorScheme.textWeak,
            style = Theme.typography.header
        )

        TabRow(
            modifier = modifier
                .fillMaxWidth()
                .clip(shape = CircleShape),
            selectedTabIndex = selectedTabIndex,
            containerColor = Theme.colorScheme.inputBackground,
            contentColor = Theme.colorScheme.backgroundDropdown,
            indicator = {},
            divider = {}
        ) {
            tabs.forEachIndexed { index, tab ->
                val isSelected = selectedTabIndex == index

                Tab(
                    modifier = Modifier
                        .padding(all = ThemeSpacing.ExtraSmall)
                        .clip(shape = RoundedCornerShape(size = ThemeRadius.Large))
                        .applyIf(
                            condition = isSelected,
                            ifTrue = { backgroundDropdownMenu() }
                        ),
                    selected = isSelected,
                    onClick = { onTabSelected(index) },
                    text = {
                        Text(
                            text = tab,
                            color = Theme.colorScheme.textNorm,
                            style = Theme.typography.body1Regular
                        )
                    }
                )
            }
        }
    }
}
