/*
 * Copyright (c) 2025 Proton AG
 * This file is part of Proton AG and Proton Authenticator.
 *
 * Proton Authenticator is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Authenticator is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Authenticator.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.authenticator.shared.ui.domain.components.lists

import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.LazyListState
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import proton.android.authenticator.shared.ui.domain.models.UiDraggableItem
import proton.android.authenticator.shared.ui.domain.modifiers.applyIf
import proton.android.authenticator.shared.ui.domain.theme.Theme
import proton.android.authenticator.shared.ui.domain.theme.ThemePadding
import proton.android.authenticator.shared.ui.domain.theme.ThemeRadius
import proton.android.authenticator.shared.ui.domain.theme.ThemeSpacing
import proton.android.authenticator.shared.ui.domain.theme.ThemeThickness
import sh.calvin.reorderable.ReorderableItem
import sh.calvin.reorderable.rememberReorderableLazyListState

@Composable
fun DraggableVerticalList(
    draggableItems: List<UiDraggableItem>,
    listState: LazyListState,
    onSorted: (Map<String, Int>) -> Unit,
    needsBottomExtraSpace: Boolean,
    modifier: Modifier = Modifier,
    contentPadding: PaddingValues = PaddingValues(all = ThemePadding.None),
    reverseLayout: Boolean = false,
    verticalArrangement: Arrangement.Vertical = if (!reverseLayout) Arrangement.Top else Arrangement.Bottom,
    isDragEnabled: Boolean = true,
    highlightColor: Color = Theme.colorScheme.inputBorderFocused
) {
    val bottomSpace = remember(key1 = needsBottomExtraSpace) {
        if (needsBottomExtraSpace) ThemeSpacing.ExtraLarge.plus(ThemeSpacing.Medium)
        else ThemeSpacing.Small
    }

    var items by remember(key1 = draggableItems) { mutableStateOf(draggableItems) }

    var selectedItemId by remember { mutableStateOf<String?>(null) }

    val reorderableLazyListState = rememberReorderableLazyListState(
        lazyListState = listState
    ) { from, to ->
        items = items.toMutableList().apply {
            add(to.index, removeAt(from.index))
        }
    }

    LazyColumn(
        modifier = modifier,
        state = listState,
        contentPadding = contentPadding,
        verticalArrangement = verticalArrangement
    ) {
        itemsIndexed(
            items = items,
            key = { _, item -> item.id }
        ) { index, item ->
            ReorderableItem(
                state = reorderableLazyListState,
                key = item.id
            ) {
                Box(
                    modifier = Modifier
                        .applyIf(
                            condition = selectedItemId == item.id,
                            ifTrue = {
                                border(
                                    shape = RoundedCornerShape(size = ThemeRadius.MediumSmall),
                                    width = ThemeThickness.Small,
                                    color = highlightColor
                                )
                            }
                        )
                        .longPressDraggableHandle(
                            enabled = isDragEnabled,
                            onDragStarted = {
                                selectedItemId = item.id
                            },
                            onDragStopped = {
                                selectedItemId = null

                                items.mapIndexed { index, uiDraggableItem ->
                                    uiDraggableItem.id to index
                                }
                                    .toMap()
                                    .also(onSorted)
                            }
                        )
                ) {
                    item.content()
                }
            }
        }

        item {
            Spacer(modifier = Modifier.height(height = bottomSpace))
        }
    }
}
