/*
 * Copyright (c) 2025 Proton AG
 * This file is part of Proton AG and Proton Authenticator.
 *
 * Proton Authenticator is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Authenticator is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Authenticator.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.authenticator.features.settings.master.presentation

import android.content.Context
import androidx.annotation.StringRes
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.flowOf
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import proton.android.authenticator.business.anonymous.data.domain.AnonymousData
import proton.android.authenticator.business.applock.domain.AppLockState
import proton.android.authenticator.business.biometrics.application.authentication.AuthenticateBiometricReason
import proton.android.authenticator.business.settings.domain.SettingsAppLockType
import proton.android.authenticator.business.settings.domain.SettingsDigitType
import proton.android.authenticator.business.settings.domain.SettingsSearchBarType
import proton.android.authenticator.business.settings.domain.SettingsSortingType
import proton.android.authenticator.business.settings.domain.SettingsThemeType
import proton.android.authenticator.features.settings.master.R
import proton.android.authenticator.features.settings.master.usecases.ObserveAnonymousDataUseCase
import proton.android.authenticator.features.settings.master.usecases.ObserveUninstalledProtonApps
import proton.android.authenticator.features.settings.master.usecases.UpdateAnonymousDataUseCase
import proton.android.authenticator.features.shared.app.usecases.GetAppVersionNameUseCase
import proton.android.authenticator.features.shared.app.usecases.GetBuildFlavorUseCase
import proton.android.authenticator.features.shared.entries.usecases.ObserveHasEntryModelsUseCase
import proton.android.authenticator.features.shared.usecases.applock.UpdateAppLockStateUseCase
import proton.android.authenticator.features.shared.usecases.biometrics.AuthenticateBiometricUseCase
import proton.android.authenticator.features.shared.usecases.settings.ObserveSettingsUseCase
import proton.android.authenticator.features.shared.usecases.settings.UpdateSettingsUseCase
import proton.android.authenticator.features.shared.usecases.snackbars.DispatchSnackbarEventUseCase
import proton.android.authenticator.features.shared.users.usecases.ObserveIsUserAuthenticatedUseCase
import proton.android.authenticator.features.shared.users.usecases.ObserveUserUseCase
import proton.android.authenticator.shared.common.domain.answers.Answer
import proton.android.authenticator.shared.common.domain.configs.AppConfig
import proton.android.authenticator.shared.common.domain.models.SnackbarEvent
import proton.android.authenticator.shared.common.flowCombine.combineN
import proton.android.authenticator.shared.common.logs.AuthenticatorLogger
import javax.inject.Inject

@HiltViewModel
internal class SettingsMasterViewModel @Inject constructor(
    getBuildFlavorUseCase: GetBuildFlavorUseCase,
    getAppVersionNameUseCase: GetAppVersionNameUseCase,
    observeHasEntryModelsUseCase: ObserveHasEntryModelsUseCase,
    observeSettingsUseCase: ObserveSettingsUseCase,
    observeUninstalledProtonApps: ObserveUninstalledProtonApps,
    observeUserUseCase: ObserveUserUseCase,
    observeAnonymousDataUseCase: ObserveAnonymousDataUseCase,
    private val authenticateBiometricUseCase: AuthenticateBiometricUseCase,
    private val updateSettingsUseCase: UpdateSettingsUseCase,
    private val updateAppLockStateUseCase: UpdateAppLockStateUseCase,
    private val dispatchSnackbarEventUseCase: DispatchSnackbarEventUseCase,
    private val observeIsUserAuthenticatedUseCase: ObserveIsUserAuthenticatedUseCase,
    private val updateAnonymousDataUseCase: UpdateAnonymousDataUseCase,
    private val appConfig: AppConfig
) : ViewModel() {

    private val eventFlow = MutableStateFlow<SettingsMasterEvent>(value = SettingsMasterEvent.Idle)

    private val configModelFlow = combine(
        getAppVersionNameUseCase().let(::flowOf),
        getBuildFlavorUseCase().let(::flowOf),
        observeHasEntryModelsUseCase(),
        observeAnonymousDataUseCase(),
        ::SettingsMasterConfigModel
    )

    internal val stateFlow: StateFlow<SettingsMasterState> = combineN(
        eventFlow,
        configModelFlow,
        observeSettingsUseCase(),
        observeUninstalledProtonApps(),
        observeUserUseCase(),
        flowOf(appConfig)
    ) { event, configModel, settings, uninstalledProtonApps, user, appConfig ->
        SettingsMasterState.Ready(
            event = event,
            configModel = configModel,
            settings = settings,
            uninstalledProtonApps = uninstalledProtonApps,
            user = user,
            appConfig = appConfig
        )
    }.stateIn(
        scope = viewModelScope,
        started = SharingStarted.WhileSubscribed(stopTimeoutMillis = 5_000),
        initialValue = SettingsMasterState.Loading
    )

    internal fun onConsumeEvent(event: SettingsMasterEvent) {
        eventFlow.compareAndSet(expect = event, update = SettingsMasterEvent.Idle)
    }

    internal fun onToggleShareCrashReport(anonymousData: AnonymousData, newIsCrashReportEnabled: Boolean) {
        anonymousData
            .copy(isCrashReportEnabled = newIsCrashReportEnabled)
            .also(::updateAnonymousData)
    }

    internal fun onToggleShareTelemetry(anonymousData: AnonymousData, newIsTelemetryEnabled: Boolean) {
        anonymousData
            .copy(isTelemetryEnabled = newIsTelemetryEnabled)
            .also(::updateAnonymousData)
    }

    private fun updateAnonymousData(anonymousData: AnonymousData) {
        viewModelScope.launch {
            updateAnonymousDataUseCase(anonymousData = anonymousData).fold(
                onFailure = { reason ->
                    AuthenticatorLogger.w(TAG, "Failed to update anonymous data sharing: $reason")

                    dispatchSnackbarMessage(
                        messageResId = R.string.settings_snackbar_message_anonymous_data_sharing_error
                    )
                },
                onSuccess = {
                    AuthenticatorLogger.i(TAG, "Successfully updated anonymous data sharing")
                }
            )
        }
    }

    internal fun onUpdateIsPassBannerDismissed(settingsModel: SettingsMasterSettingsModel) {
        settingsModel.copy(isPassBannerDismissed = true)
            .also(::updateSettings)
    }

    internal fun onUpdateIsSyncEnabled(settingsModel: SettingsMasterSettingsModel, newIsSyncEnabled: Boolean) {
        if (newIsSyncEnabled) {
            viewModelScope.launch {
                if (observeIsUserAuthenticatedUseCase().first()) {
                    settingsModel.copy(isSyncEnabled = true).also(::updateSettings)
                    return@launch
                }

                eventFlow.update { SettingsMasterEvent.OnSyncEnabled }
            }
        } else {
            eventFlow.update { SettingsMasterEvent.OnSyncDisabled }
        }
    }

    internal fun onUpdateAppLockType(
        settingsModel: SettingsMasterSettingsModel,
        newAppLockType: SettingsAppLockType,
        context: Context
    ) {
        if (settingsModel.appLockType == newAppLockType) return

        when (newAppLockType) {
            SettingsAppLockType.None -> Triple(
                first = R.string.settings_security_lock_disable_title,
                second = R.string.settings_security_lock_disable_subtitle,
                third = proton.android.authenticator.shared.ui.R.string.action_cancel
            )

            SettingsAppLockType.Biometric -> Triple(
                first = R.string.settings_security_lock_enable_title,
                second = R.string.settings_security_lock_enable_subtitle,
                third = proton.android.authenticator.shared.ui.R.string.action_cancel
            )
        }.also { (titleResId, subtitleResId, cancelButtonResId) ->
            updateAppLockType(
                titleResId = titleResId,
                subtitleResId = subtitleResId,
                cancelButtonResId = cancelButtonResId,
                context = context,
                appLockType = newAppLockType,
                settingsModel = settingsModel
            )
        }
    }

    @Suppress("LongParameterList")
    private fun updateAppLockType(
        @StringRes titleResId: Int,
        @StringRes subtitleResId: Int,
        @StringRes cancelButtonResId: Int,
        context: Context,
        appLockType: SettingsAppLockType,
        settingsModel: SettingsMasterSettingsModel
    ) {
        viewModelScope.launch {
            authenticateBiometricUseCase(
                title = context.getString(titleResId),
                subtitle = context.getString(subtitleResId),
                cancelButton = context.getString(cancelButtonResId),
                context = context
            ).fold(
                onFailure = { reason ->
                    when (reason) {
                        AuthenticateBiometricReason.UserCanceled -> Unit

                        AuthenticateBiometricReason.Canceled,
                        AuthenticateBiometricReason.HardwareNotPresent,
                        AuthenticateBiometricReason.HardwareUnavailable,
                        AuthenticateBiometricReason.Lockout,
                        AuthenticateBiometricReason.LockoutPermanent,
                        AuthenticateBiometricReason.NegativeButton,
                        AuthenticateBiometricReason.NoBiometrics,
                        AuthenticateBiometricReason.NoDeviceCredential,
                        AuthenticateBiometricReason.NoSpace,
                        AuthenticateBiometricReason.NotEnrolled,
                        AuthenticateBiometricReason.Timeout,
                        AuthenticateBiometricReason.UnableToProcess,
                        AuthenticateBiometricReason.Unavailable,
                        AuthenticateBiometricReason.Unknown,
                        AuthenticateBiometricReason.Unsupported,
                        AuthenticateBiometricReason.Vendor,
                        AuthenticateBiometricReason.WrongContext -> {
                            dispatchSnackbarMessage(
                                messageResId = R.string.settings_snackbar_message_biometric_error
                            )
                        }
                    }
                },
                onSuccess = {
                    if (appLockType == SettingsAppLockType.Biometric) {
                        updateAppLockStateUseCase(state = AppLockState.AuthNotRequired)
                    }

                    settingsModel.copy(appLockType = appLockType)
                        .also(::updateSettings)
                }
            )
        }
    }

    internal fun onUpdateIsTapToRevealEnabled(
        settingsModel: SettingsMasterSettingsModel,
        newIsTapToRevealEnabled: Boolean
    ) {
        settingsModel.copy(isHideCodesEnabled = newIsTapToRevealEnabled)
            .also(::updateSettings)
    }

    internal fun onUpdateThemeType(settingsModel: SettingsMasterSettingsModel, newThemeType: SettingsThemeType) {
        if (settingsModel.themeType == newThemeType) return

        settingsModel.copy(themeType = newThemeType)
            .also(::updateSettings)
    }

    internal fun onUpdateSearchBarType(
        settingsModel: SettingsMasterSettingsModel,
        newSearchBarType: SettingsSearchBarType
    ) {
        if (settingsModel.searchBarType == newSearchBarType) return

        settingsModel.copy(searchBarType = newSearchBarType)
            .also(::updateSettings)
    }

    internal fun onUpdateDigitType(settingsModel: SettingsMasterSettingsModel, newDigitType: SettingsDigitType) {
        if (settingsModel.digitType == newDigitType) return

        settingsModel.copy(digitType = newDigitType)
            .also(::updateSettings)
    }

    internal fun onUpdateSortingType(settingsModel: SettingsMasterSettingsModel, newSortingType: SettingsSortingType) {
        if (settingsModel.sortingType == newSortingType) return

        settingsModel.copy(sortingType = newSortingType)
            .also(::updateSettings)
    }

    internal fun onUpdateIsCodeChangeAnimationEnabled(
        settingsModel: SettingsMasterSettingsModel,
        newIsCodeChangeAnimationEnabled: Boolean
    ) {
        settingsModel.copy(isCodeChangeAnimationEnabled = newIsCodeChangeAnimationEnabled)
            .also(::updateSettings)
    }

    private fun updateSettings(newSettingsModel: SettingsMasterSettingsModel) {
        viewModelScope.launch {
            updateSettingsUseCase(newSettingsModel.asSettings()).also { answer ->
                when (answer) {
                    is Answer.Failure -> dispatchSnackbarMessage(
                        messageResId = R.string.settings_snackbar_message_update_error
                    )

                    is Answer.Success -> Unit
                }
            }
        }
    }

    private suspend fun dispatchSnackbarMessage(@StringRes messageResId: Int) {
        SnackbarEvent(messageResId = messageResId).also { snackbarEvent ->
            dispatchSnackbarEventUseCase(snackbarEvent)
        }
    }

    private companion object {

        private const val TAG = "SettingsMasterViewModel"

    }

}
