/*
 * Copyright (c) 2025 Proton AG
 * This file is part of Proton AG and Proton Authenticator.
 *
 * Proton Authenticator is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Proton Authenticator is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Proton Authenticator.  If not, see <https://www.gnu.org/licenses/>.
 */

package proton.android.authenticator.features.backups.master.presentation

import android.net.Uri
import androidx.annotation.StringRes
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.collections.immutable.toPersistentList
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import proton.android.authenticator.business.backups.domain.BackupFrequencyType
import proton.android.authenticator.features.backups.master.R
import proton.android.authenticator.features.shared.entries.presentation.EntryModel
import proton.android.authenticator.features.shared.entries.usecases.ObserveEntryModelsUseCase
import proton.android.authenticator.features.shared.usecases.backups.GenerateBackupUseCase
import proton.android.authenticator.features.shared.usecases.backups.ObserveBackupUseCase
import proton.android.authenticator.features.shared.usecases.backups.UpdateBackupUseCase
import proton.android.authenticator.features.shared.usecases.snackbars.DispatchSnackbarEventUseCase
import proton.android.authenticator.shared.common.domain.answers.Answer
import proton.android.authenticator.shared.common.domain.models.SnackbarEvent
import proton.android.authenticator.shared.common.logs.AuthenticatorLogger
import javax.inject.Inject

@HiltViewModel
internal class BackupsMasterViewModel @Inject constructor(
    observeBackupUseCase: ObserveBackupUseCase,
    observeEntryModelsUseCase: ObserveEntryModelsUseCase,
    private val generateBackupUseCase: GenerateBackupUseCase,
    private val updateBackupUseCase: UpdateBackupUseCase,
    private val dispatchSnackbarEventUseCase: DispatchSnackbarEventUseCase
) : ViewModel() {

    private val backupModel: BackupMasterModel
        get() = stateFlow.value.backupModel

    private val backupFlow = observeBackupUseCase()
    private val entryModelsFlow = observeEntryModelsUseCase()
        .map(List<EntryModel>::toPersistentList)

    private val eventFlow = MutableStateFlow<BackupMasterEvent>(value = BackupMasterEvent.Idle)

    internal val stateFlow: StateFlow<BackupsMasterState> = combine(
        entryModelsFlow,
        eventFlow,
        backupFlow,
        ::BackupsMasterState
    ).stateIn(
        scope = viewModelScope,
        started = SharingStarted.WhileSubscribed(stopTimeoutMillis = 5_000),
        initialValue = BackupsMasterState.Initial
    )
    internal fun onConsumeEvent(event: BackupMasterEvent) {
        eventFlow.compareAndSet(expect = event, update = BackupMasterEvent.Idle)
    }

    internal fun onDisableBackup() {
        if (!backupModel.isEnabled) return

        backupModel.copy(
            isEnabled = false,
            directoryUri = Uri.EMPTY,
            encryptedPassword = null
        ).also(::updateBackup)
    }

    internal fun onFolderPicked(uri: Uri) {
        if (backupModel.directoryUri == uri) {
            AuthenticatorLogger.i(TAG, "Backup folder selection aborted by user")
            return
        }

        eventFlow.update { BackupMasterEvent.OnBackupPassword(uri = uri.toString()) }
    }

    internal fun onUpdateFrequencyType(newFrequencyType: BackupFrequencyType) {
        if (backupModel.frequencyType == newFrequencyType) return

        backupModel.copy(frequencyType = newFrequencyType)
            .also(::updateBackup)
    }

    internal fun onCreateBackup(entryModels: List<EntryModel>) {
        viewModelScope.launch {
            generateBackupUseCase(entryModels).fold(
                onFailure = { reason ->
                    AuthenticatorLogger.w(TAG, "Failed to generate backup: $reason")

                    onDisableBackup()

                    eventFlow.update { BackupMasterEvent.OnBackupError(reason = reason) }
                },
                onSuccess = {
                    AuthenticatorLogger.i(TAG, "Successfully generated backup")

                    dispatchSnackbarMessage(messageResId = R.string.backups_snackbar_message_backup_success)
                }
            )
        }
    }
    private fun updateBackup(newBackupMasterBackup: BackupMasterModel) {
        viewModelScope.launch {
            updateBackupUseCase(newBackupMasterBackup.asBackup()).let { answer ->
                when (answer) {
                    is Answer.Failure -> dispatchSnackbarMessage(
                        messageResId = R.string.backups_snackbar_message_update_error
                    )

                    is Answer.Success -> Unit
                }
            }
        }
    }

    private suspend fun dispatchSnackbarMessage(@StringRes messageResId: Int) {
        SnackbarEvent(messageResId = messageResId).also { snackbarEvent ->
            dispatchSnackbarEventUseCase(snackbarEvent)
        }
    }

    private companion object {

        private const val TAG = "BackupsMasterViewModel"

    }

}
