/*
Copyright 2022 Andreas Ritter (www.wolfbeargames.de)

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and
associated documentation files (the "Software"), to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge, publish, distribute,
sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT
NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT
OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/
package de.wolfbeargames.geolocationplugin

import android.Manifest
import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.content.pm.PackageManager
import android.location.Location
import android.location.LocationManager
import android.location.LocationRequest
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.util.Log
import android.view.View
import android.widget.Toast
import androidx.annotation.RequiresApi
import androidx.core.content.ContextCompat
import org.godotengine.godot.Dictionary
import org.godotengine.godot.Godot
import org.godotengine.godot.plugin.GodotPlugin
import org.godotengine.godot.plugin.SignalInfo
import org.godotengine.godot.plugin.UsedByGodot

class GeolocationPlugin(godot: Godot) : GodotPlugin(godot) {
    override fun getPluginName(): String {
        return "Geolocation"
    }

    @UsedByGodot
    private fun helloWorld() {
        runOnUiThread {
            Toast.makeText(activity, "Hello World", Toast.LENGTH_LONG).show()
            Log.v(pluginName, "Hello World")
            send_log_signal("m hello world function")
        }
    }

    //@RequiresApi(Build.VERSION_CODES.S)
    @SuppressLint("MissingPermission")
    private fun requestAndroidLocation() {
        val locationManager = activity?.getSystemService(Context.LOCATION_SERVICE) as? LocationManager
        val request = LocationRequest.Builder(5000).setQuality(LocationRequest.QUALITY_HIGH_ACCURACY).build()
        locationManager!!.getCurrentLocation(LocationManager.FUSED_PROVIDER, request, null,
            {  },
            {  })
        val lastLocation = locationManager.getLastKnownLocation(LocationManager.FUSED_PROVIDER)
        if (lastLocation == null) {
            runOnUiThread {
                Toast.makeText(activity, "Location NULL - Try Again", Toast.LENGTH_LONG).show()
            }
        }else{
            sendLocationUpdate(lastLocation)
        }
    }

    //region Enum definition
    enum class GeolocationAuthorizationStatus(val value: Int) {
        PERMISSION_STATUS_UNKNOWN(1 shl 0),
        PERMISSION_STATUS_DENIED(1 shl 1),
        PERMISSION_STATUS_ALLOWED(1 shl 2)
    }

    enum class GeolocationDesiredAccuracyConstants(val value: Int) {
        ACCURACY_BEST_FOR_NAVIGATION(1 shl 0),
        ACCURACY_BEST(1 shl 1),
        ACCURACY_NEAREST_TEN_METERS(1 shl 2),
        ACCURACY_HUNDRED_METERS(1 shl 3),
        ACCURACY_KILOMETER(1 shl 40),
        ACCURACY_THREE_KILOMETER(1 shl 5),
        ACCURACY_REDUCED(1 shl 6),
    }

    enum class GeolocationErrorCodes(val value: Int) {
        ERROR_DENIED(1 shl 0),
        ERROR_NETWORK(1 shl 1),
        ERROR_HEADING_FAILURE(1 shl 2),
        ERROR_LOCATION_UNKNOWN(1 shl 3),
        ERROR_TIMEOUT(1 shl 4),
        ERROR_UNSUPPORTED(1 shl 5),
        ERROR_LOCATION_DISABLED(1 shl 6),
        ERROR_UNKNOWN(1 shl 7),
    }

    //endregion

    //region Signal Definition
    private val _logSignal =
        SignalInfo("log", String::class.java, Float::class.javaObjectType)
    private val _errorSignal =
        SignalInfo("error", Int::class.javaObjectType)
    private val _locationUpdateSignal =
        SignalInfo("location_update", Dictionary::class.java)
    private val _authorizationChangedSignal =
        SignalInfo("authorization_changed", Int::class.javaObjectType)
    private val _headingUpdateSignal =
        SignalInfo("heading_update", Dictionary::class.java)
    private val _locationCapabiltityResult =
        SignalInfo(
            "location_capability_result",
            Boolean::class.javaObjectType
        )

    override fun getPluginSignals(): Set<SignalInfo> {
        return setOf(
            _logSignal,
            _errorSignal,
            _locationUpdateSignal,
            _authorizationChangedSignal,
            _headingUpdateSignal,
            _locationCapabiltityResult
        )
    }
    //endregion

    //region Initialization
    // references
    //private var _locationCallback: LocationCallback? = null
    //private var _locationOneTimeCallback: LocationCallback? = null
    //private var _locationManager: FusedLocationProviderClient? = null

    private var _handler: Handler? = null
    private var _timeoutRunnable: Runnable? = null

    // state
    private var _locationUpdatesActive: Boolean = false
    private var _hasCoarsePermission: Boolean = false
    private var _hasFinePermission: Boolean = false
    private var _failureTimeoutRunning: Boolean = false
    private var _locationUpdatesPaused: Boolean = false

    // data storage
    private var _lastLocationData = Dictionary()

    // settings
    private var _distanceFilter: Float = 0f
    //private var _desiredAccuracy: Int = Priority.PRIORITY_HIGH_ACCURACY
    private var _desiredUpdateInterval: Long = 1 // in seconds
    private var _maxWaitTime: Long = 1 // in seconds
    private var _returnStringCoordinates: Boolean = true
    private var _sendDebugLogSignal: Boolean = false
    private var _autoCheckLocationCapability: Boolean = false
    private var _failureTimeout: Long = 20 // also set in constructor
    private var _useFailureTimeout: Boolean = true
    private var _locationProvider: String = "FUSED"

    // list of all supported methods
    private val _supportedMethods: Array<String> = arrayOf("authorization_status",
        "allows_full_accuracy", "can_request_permissions","is_updating_location",
        "set_distance_filter","set_desired_accuracy","set_return_string_coordinates",
        "request_location","start_updating_location","stop_updating_location",
        "request_location_capabilty", "set_debug_log_signal","set_failure_timeout",
        "should_show_permission_requirement_explanation", "set_update_interval", "set_max_wait_time",
        "set_auto_check_location_capability","should_check_location_capability")

    override fun onMainCreate(activity: Activity?): View? {

        //set_failure_timeout(20)

        _handler = Handler(Looper.getMainLooper())
        _timeoutRunnable = Runnable {
            StopFailureTimeout()
        //    _locationManager!!.removeLocationUpdates(_locationCallback!!)
        //    _locationManager!!.removeLocationUpdates(_locationOneTimeCallback!!)
            _locationUpdatesActive = false
            send_error_signal(GeolocationErrorCodes.ERROR_TIMEOUT.value)
        }

        //_locationCallback = object : LocationCallback() {
        //    override fun onLocationResult(locationResult: LocationResult) {
        //        if (_failureTimeoutRunning) StopFailureTimeout()
        //        send_log_signal("m location_update UPDATE location")
        //        for (location in locationResult.locations) {
        //            sendLocationUpdate(location)
        //        }
        //    }
        //}

        //_locationOneTimeCallback = object : LocationCallback() {
        //    override fun onLocationResult(locationResult: LocationResult) {
        //        if (_failureTimeoutRunning) StopFailureTimeout()
        //        _locationManager!!.removeLocationUpdates(this)
        //        send_log_signal("m request_location ONE location")
        //        sendLocationUpdate(locationResult.lastLocation!!)
        //    }
        //}

        return super.onMainCreate(activity)
    }


    private fun StartFailureTimeout() {
        _failureTimeoutRunning = true
        //_handler!!.postDelayed(_timeoutRunnable!!, _failureTimeout * 1000)
    }

    private fun StopFailureTimeout() {
        _failureTimeoutRunning = false
        //_handler!!.removeCallbacks(_timeoutRunnable!!)
    }

    private fun hasAppropriateLocationCapability(continueWith: ((request: LocationRequest?) -> Unit)? = null) {
        val locationManager = activity?.getSystemService(Context.LOCATION_SERVICE) as? LocationManager
        val gpsEnabled = locationManager?.isProviderEnabled(LocationManager.GPS_PROVIDER)?: false
        val networkEnabled = locationManager?.isProviderEnabled(LocationManager.NETWORK_PROVIDER)?: false

        if (gpsEnabled || networkEnabled) {
            send_log_signal("m location_capability result: TRUE")
            send_location_capability_signal(true)

            if (continueWith!= null) {
                continueWith(null)
            }
        } else {
            send_log_signal("m location_capability result: FALSE")
            send_location_capability_signal(false)

            if (continueWith!= null) {
                send_error_signal(GeolocationErrorCodes.ERROR_LOCATION_DISABLED.value)
            }
        }
    }

    private fun hasLocationPermissions(): Boolean {
        val fineLocationPermission =
            ContextCompat.checkSelfPermission(activity!!, Manifest.permission.ACCESS_FINE_LOCATION)
        val coarseLocationPermission = ContextCompat.checkSelfPermission(
            activity!!,
            Manifest.permission.ACCESS_COARSE_LOCATION
        )

        return fineLocationPermission == PackageManager.PERMISSION_GRANTED || coarseLocationPermission == PackageManager.PERMISSION_GRANTED
    }

    //@SuppressLint("MissingPermission")
    //private fun requestAndroidLocation(request :LocationRequest? = null) {
    //    val locationManager = context.getSystemService(Context.LOCATION_SERVICE)
    //    locationManager!!.getLastKnownLocation(
    //        LocationManager.GPS_PROVIDER, // FUSED_PROVIDER
    //        _locationOneTimeCallback!!,
    //        Looper.getMainLooper()
    //    ).addOnFailureListener { // don't know if this works and has any effect
    //        send_error_signal(GeolocationErrorCodes.ERROR_LOCATION_UNKNOWN.value)
    //    }

    //    if(_useFailureTimeout) StartFailureTimeout()

    //}

    //@SuppressLint("MissingPermission")
    //private fun RequestLocationInternal(request :LocationRequest? = null) {
        // workaround because getCurrentLocation does not work ("No virtual method getCurrentLocation" Error):
        // request location updates and stop as soon as we got one location
        //val locationManager = context.getSystemService(Context.LOCATION_SERVICE) as LocationManager
        //locationManager.getLastKnownLocation(LocationManager.FUSED_PROVIDER)
        //_locationManager!!.requestLocationUpdates(
        //    LocationRequest.create().setPriority(_desiredAccuracy).setMaxWaitTime(0),
        //    _locationOneTimeCallback!!,
        //    Looper.getMainLooper()
        //).addOnFailureListener { // don't know if this works and has any effect
        //    send_error_signal(GeolocationErrorCodes.ERROR_LOCATION_UNKNOWN.value)
        //}

        //if(_useFailureTimeout) StartFailureTimeout()
    //}

    private fun sendLocationUpdate(location: Location) {
        val locationData = Dictionary()
        locationData["latitude"] = location.latitude
        locationData["longitude"] = location.longitude
        locationData["accuracy"] = location.accuracy
        locationData["altitude"] = location.altitude
        locationData["bearing"] = location.bearing
        locationData["speed"] = location.speed
        locationData["timestamp"] = location.time

        //locationData["altitude_accuracy"] = location.verticalAccuracyMeters
        //locationData["speed_accuracy"] = location.speedAccuracyMetersPerSecond
        //locationData["bearing_accuracy"] = location.bearingAccuracyDegrees

        // Add other location properties as needed

        emitSignal("location_update", locationData)
    }

    private fun updateAuthorizationStatus() {
        val statusFine =
            ContextCompat.checkSelfPermission(activity!!, Manifest.permission.ACCESS_FINE_LOCATION)
        val statusCoarse = ContextCompat.checkSelfPermission(
            activity!!,
            Manifest.permission.ACCESS_COARSE_LOCATION
        )

        _hasCoarsePermission = (statusCoarse == PackageManager.PERMISSION_GRANTED)
        _hasFinePermission = (statusFine == PackageManager.PERMISSION_GRANTED)
    }

    //region Authorization, permission and status

    // unused on Android
    @UsedByGodot
    fun request_permission() {
        // not implemented (use Godot mechanism to ask for permissions
        send_log_signal("m request_permission NO EFFECT")
        //send_error_signal(GeolocationErrorCodes.ERROR_UNSUPPORTED.value)
    }

    @UsedByGodot
    fun authorization_status(): Int { // Enum Geolocation::GeolocationAuthorizationStatus
        send_log_signal("m authorization_status")

        updateAuthorizationStatus()

        if (_hasCoarsePermission || _hasFinePermission)
            return GeolocationAuthorizationStatus.PERMISSION_STATUS_ALLOWED.value

        return GeolocationAuthorizationStatus.PERMISSION_STATUS_DENIED.value
    }

    @UsedByGodot
    fun should_check_location_capability(): Boolean {
        return !_autoCheckLocationCapability
    }

    @UsedByGodot
    fun request_location_capabilty() {
        send_log_signal("m location_capabilty")
        hasAppropriateLocationCapability()
    }

    @SuppressLint("MissingPermission")
    @UsedByGodot
    fun request_location() {
        send_log_signal("m request_location")
        if (!hasLocationPermissions()) return
        send_log_signal("m location api still being worked on, cannot complete request")
        // check GPS capability and continue to get location
        if(_autoCheckLocationCapability){
            hasAppropriateLocationCapability()
        } else {
            requestAndroidLocation()
        }
    }

    @UsedByGodot
    fun set_debug_log_signal(send: Boolean) {
        _sendDebugLogSignal = send
        send_log_signal("m set_debug_log_signal $send")
    }

    @UsedByGodot
    fun set_location_provider(send: String) {
        _locationProvider = send
        send_log_signal("m set_location_provider $send")
    }

    //region Signal sender methods
    private fun send_log_signal(message: String, number: Float = 0f) {
        if (!_sendDebugLogSignal) return;
        emitSignal(_logSignal.name, message, number)
    }

    private fun send_error_signal(errorCode: Int) {
        send_log_signal("m send_error_signal",errorCode.toFloat())
        emitSignal(_errorSignal.name, errorCode)
    }

    private fun send_location_update_signal(locationData: Dictionary) {
        emitSignal(_locationUpdateSignal.name, locationData)
    }

    private fun send_location_capability_signal(capable: Boolean) {
        emitSignal(_locationCapabiltityResult.name, capable)
    }

}