/*
 * Copyright (C) 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.inputmethod.keyboard.internal;

import java.util.HashMap;
import java.util.Locale;

/**
 * !!!!! DO NOT EDIT THIS FILE !!!!!
 *
 * This file is generated by tools/make-keyboard-text. The base template file is
 *   tools/make-keyboard-text/res/src/com/android/inputmethod/keyboard/internal/
 *   KeyboardTextsTable.tmpl
 *
 * This file must be updated when any text resources in keyboard layout files have been changed.
 * These text resources are referred as "!text/<resource_name>" in keyboard XML definitions,
 * and should be defined in
 *   tools/make-keyboard-text/res/values-<locale>/donottranslate-more-keys.xml
 *
 * To update this file, please run the following commands.
 *   $ cd $ANDROID_BUILD_TOP
 *   $ mmm packages/inputmethods/LatinIME/tools/make-keyboard-text
 *   $ make-keyboard-text -java packages/inputmethods/LatinIME/java
 *
 * The updated source file will be generated to the following path (this file).
 *   packages/inputmethods/LatinIME/java/src/com/android/inputmethod/keyboard/internal/
 *   KeyboardTextsTable.java
 */
public final class KeyboardTextsTable {
    // Name to index map.
    private static final HashMap<String, Integer> sNameToIndexesMap = new HashMap<>();
    // Locale to texts table map.
    private static final HashMap<String, String[]> sLocaleToTextsTableMap = new HashMap<>();
    // TODO: Remove this variable after debugging.
    // Texts table to locale maps.
    private static final HashMap<String[], String> sTextsTableToLocaleMap = new HashMap<>();

    public static String getText(final String name, final String[] textsTable) {
        final Integer indexObj = sNameToIndexesMap.get(name);
        if (indexObj == null) {
            throw new RuntimeException("Unknown text name=" + name + " locale="
                    + sTextsTableToLocaleMap.get(textsTable));
        }
        final int index = indexObj;
        final String text = (index < textsTable.length) ? textsTable[index] : null;
        if (text != null) {
            return text;
        }
        // Sanity check.
        if (index >= 0 && index < TEXTS_DEFAULT.length) {
            return TEXTS_DEFAULT[index];
        }
        // Throw exception for debugging purpose.
        throw new RuntimeException("Illegal index=" + index + " for name=" + name
                + " locale=" + sTextsTableToLocaleMap.get(textsTable));
    }

    public static String[] getTextsTable(final Locale locale) {
        final String localeKey = locale.toString();
        if (sLocaleToTextsTableMap.containsKey(localeKey)) {
            return sLocaleToTextsTableMap.get(localeKey);
        }
        final String languageKey = locale.getLanguage();
        if (sLocaleToTextsTableMap.containsKey(languageKey)) {
            return sLocaleToTextsTableMap.get(languageKey);
        }
        return TEXTS_DEFAULT;
    }

    private static final String[] NAMES = {
    //  /* index:histogram */ "name",
        /*   0:15 */ "keylabel_to_alpha",
        /*   1:13 */ "additional_morekeys_symbols_1",
        /*   2:13 */ "additional_morekeys_symbols_2",
        /*   3:13 */ "additional_morekeys_symbols_3",
        /*   4:13 */ "additional_morekeys_symbols_4",
        /*   5:13 */ "additional_morekeys_symbols_5",
        /*   6:13 */ "additional_morekeys_symbols_6",
        /*   7:13 */ "additional_morekeys_symbols_7",
        /*   8:13 */ "additional_morekeys_symbols_8",
        /*   9:13 */ "additional_morekeys_symbols_9",
        /*  10:13 */ "additional_morekeys_symbols_0",
        /*  11:12 */ "keylabel_to_symbol",
        /*  12:11 */ "keyspec_symbols_1",
        /*  13:11 */ "keyspec_symbols_2",
        /*  14:11 */ "keyspec_symbols_3",
        /*  15:11 */ "keyspec_symbols_4",
        /*  16:11 */ "keyspec_symbols_5",
        /*  17:11 */ "keyspec_symbols_6",
        /*  18:11 */ "keyspec_symbols_7",
        /*  19:11 */ "keyspec_symbols_8",
        /*  20:11 */ "keyspec_symbols_9",
        /*  21:11 */ "keyspec_symbols_0",
        /*  22: 6 */ "keyspec_currency",
        /*  23: 4 */ "morekeys_period",
        /*  24: 3 */ "keyspec_comma",
        /*  25: 3 */ "keyspec_period",
        /*  26: 3 */ "keyspec_tablet_period",
        /*  27: 3 */ "morekeys_tablet_period",
        /*  28: 2 */ "morekeys_a",
        /*  29: 2 */ "morekeys_e",
        /*  30: 2 */ "morekeys_i",
        /*  31: 2 */ "morekeys_o",
        /*  32: 2 */ "morekeys_u",
        /*  33: 2 */ "morekeys_s",
        /*  34: 2 */ "morekeys_n",
        /*  35: 2 */ "morekeys_c",
        /*  36: 1 */ "morekeys_y",
        /*  37: 1 */ "morekeys_d",
        /*  38: 1 */ "morekeys_r",
        /*  39: 1 */ "morekeys_t",
        /*  40: 1 */ "morekeys_z",
        /*  41: 1 */ "morekeys_k",
        /*  42: 1 */ "morekeys_l",
        /*  43: 1 */ "morekeys_g",
        /*  44: 1 */ "morekeys_h",
        /*  45: 1 */ "morekeys_j",
        /*  46: 1 */ "morekeys_w",
        /*  47: 1 */ "morekeys_currency_dollar",
        /*  48: 1 */ "keyspec_symbols_semicolon",
        /*  49: 1 */ "label_go_key",
        /*  50: 1 */ "label_send_key",
        /*  51: 1 */ "label_next_key",
        /*  52: 1 */ "label_done_key",
        /*  53: 1 */ "label_search_key",
        /*  54: 1 */ "label_previous_key",
        /*  55: 1 */ "label_pause_key",
        /*  56: 1 */ "label_wait_key",
        /*  57: 0 */ "morekeys_v",
        /*  58: 0 */ "morekeys_q",
        /*  59: 0 */ "morekeys_x",
        /*  60: 0 */ "keyspec_q",
        /*  61: 0 */ "keyspec_w",
        /*  62: 0 */ "keyspec_y",
        /*  63: 0 */ "keyspec_x",
        /*  64: 0 */ "keyspec_nordic_row1_11",
        /*  65: 0 */ "keyspec_nordic_row2_10",
        /*  66: 0 */ "keyspec_nordic_row2_11",
        /*  67: 0 */ "morekeys_nordic_row2_10",
        /*  68: 0 */ "morekeys_nordic_row2_11",
        /*  69: 0 */ "keyspec_east_slavic_row1_9",
        /*  70: 0 */ "keyspec_east_slavic_row2_2",
        /*  71: 0 */ "keyspec_east_slavic_row2_11",
        /*  72: 0 */ "keyspec_east_slavic_row3_5",
        /*  73: 0 */ "morekeys_east_slavic_row2_2",
        /*  74: 0 */ "morekeys_east_slavic_row2_11",
        /*  75: 0 */ "morekeys_cyrillic_u",
        /*  76: 0 */ "morekeys_cyrillic_ka",
        /*  77: 0 */ "morekeys_cyrillic_en",
        /*  78: 0 */ "morekeys_cyrillic_ghe",
        /*  79: 0 */ "morekeys_cyrillic_a",
        /*  80: 0 */ "morekeys_cyrillic_o",
        /*  81: 0 */ "morekeys_cyrillic_i",
        /*  82: 0 */ "morekeys_cyrillic_ie",
        /*  83: 0 */ "morekeys_cyrillic_soft_sign",
        /*  84: 0 */ "keyspec_south_slavic_row1_6",
        /*  85: 0 */ "keyspec_south_slavic_row2_11",
        /*  86: 0 */ "keyspec_south_slavic_row3_1",
        /*  87: 0 */ "keyspec_south_slavic_row3_8",
        /*  88: 0 */ "keyspec_swiss_row1_11",
        /*  89: 0 */ "keyspec_swiss_row2_10",
        /*  90: 0 */ "keyspec_swiss_row2_11",
        /*  91: 0 */ "morekeys_swiss_row1_11",
        /*  92: 0 */ "morekeys_swiss_row2_10",
        /*  93: 0 */ "morekeys_swiss_row2_11",
        /*  94: 0 */ "single_quotes",
        /*  95: 0 */ "double_quotes",
        /*  96: 0 */ "single_angle_quotes",
        /*  97: 0 */ "double_angle_quotes",
        /*  98: 0 */ "morekeys_currency_generic",
        /*  99: 0 */ "morekeys_punctuation",
        /* 100: 0 */ "morekeys_tablet_punctuation",
        /* 101: 0 */ "keyspec_spanish_row2_10",
        /* 102: 0 */ "morekeys_star",
        /* 103: 0 */ "morekeys_bullet",
        /* 104: 0 */ "morekeys_plus",
        /* 105: 0 */ "morekeys_left_parenthesis",
        /* 106: 0 */ "morekeys_right_parenthesis",
        /* 107: 0 */ "morekeys_less_than",
        /* 108: 0 */ "morekeys_greater_than",
        /* 109: 0 */ "morekeys_arabic_diacritics",
        /* 110: 0 */ "morekeys_symbols_1",
        /* 111: 0 */ "morekeys_symbols_2",
        /* 112: 0 */ "morekeys_symbols_3",
        /* 113: 0 */ "morekeys_symbols_4",
        /* 114: 0 */ "morekeys_symbols_5",
        /* 115: 0 */ "morekeys_symbols_6",
        /* 116: 0 */ "morekeys_symbols_7",
        /* 117: 0 */ "morekeys_symbols_8",
        /* 118: 0 */ "morekeys_symbols_9",
        /* 119: 0 */ "morekeys_symbols_0",
        /* 120: 0 */ "keyspec_left_parenthesis",
        /* 121: 0 */ "keyspec_right_parenthesis",
        /* 122: 0 */ "keyspec_left_square_bracket",
        /* 123: 0 */ "keyspec_right_square_bracket",
        /* 124: 0 */ "keyspec_left_curly_bracket",
        /* 125: 0 */ "keyspec_right_curly_bracket",
        /* 126: 0 */ "keyspec_less_than",
        /* 127: 0 */ "keyspec_greater_than",
        /* 128: 0 */ "keyspec_less_than_equal",
        /* 129: 0 */ "keyspec_greater_than_equal",
        /* 130: 0 */ "keyspec_left_double_angle_quote",
        /* 131: 0 */ "keyspec_right_double_angle_quote",
        /* 132: 0 */ "keyspec_left_single_angle_quote",
        /* 133: 0 */ "keyspec_right_single_angle_quote",
        /* 134: 0 */ "keyspec_tablet_comma",
        /* 135: 0 */ "keyhintlabel_tablet_comma",
        /* 136: 0 */ "morekeys_tablet_comma",
        /* 137: 0 */ "keyhintlabel_period",
        /* 138: 0 */ "keyhintlabel_tablet_period",
        /* 139: 0 */ "keyspec_symbols_question",
        /* 140: 0 */ "keyspec_symbols_percent",
        /* 141: 0 */ "morekeys_exclamation",
        /* 142: 0 */ "morekeys_question",
        /* 143: 0 */ "morekeys_symbols_semicolon",
        /* 144: 0 */ "morekeys_symbols_percent",
        /* 145: 0 */ "morekeys_am_pm",
        /* 146: 0 */ "keyspec_settings",
        /* 147: 0 */ "keyspec_shortcut",
        /* 148: 0 */ "keyspec_action_next",
        /* 149: 0 */ "keyspec_action_previous",
        /* 150: 0 */ "keylabel_to_more_symbol",
        /* 151: 0 */ "keylabel_tablet_to_more_symbol",
        /* 152: 0 */ "keylabel_to_phone_numeric",
        /* 153: 0 */ "keylabel_to_phone_symbols",
        /* 154: 0 */ "keylabel_time_am",
        /* 155: 0 */ "keylabel_time_pm",
        /* 156: 0 */ "keyspec_popular_domain",
        /* 157: 0 */ "morekeys_popular_domain",
        /* 158: 0 */ "keyspecs_left_parenthesis_more_keys",
        /* 159: 0 */ "keyspecs_right_parenthesis_more_keys",
        /* 160: 0 */ "single_laqm_raqm",
        /* 161: 0 */ "single_raqm_laqm",
        /* 162: 0 */ "double_laqm_raqm",
        /* 163: 0 */ "double_raqm_laqm",
        /* 164: 0 */ "single_lqm_rqm",
        /* 165: 0 */ "single_9qm_lqm",
        /* 166: 0 */ "single_9qm_rqm",
        /* 167: 0 */ "single_rqm_9qm",
        /* 168: 0 */ "double_lqm_rqm",
        /* 169: 0 */ "double_9qm_lqm",
        /* 170: 0 */ "double_9qm_rqm",
        /* 171: 0 */ "double_rqm_9qm",
        /* 172: 0 */ "morekeys_single_quote",
        /* 173: 0 */ "morekeys_double_quote",
        /* 174: 0 */ "morekeys_tablet_double_quote",
        /* 175: 0 */ "keyspec_emoji_action_key",
    };

    private static final String EMPTY = "";

    /* Default texts */
    private static final String[] TEXTS_DEFAULT = {
        // Label for "switch to alphabetic" key.
        /* keylabel_to_alpha */ "ABC",
        /* additional_morekeys_symbols_1 ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ additional_morekeys_symbols_0 */
        // Label for "switch to symbols" key.
        /* keylabel_to_symbol */ "?123",
        /* keyspec_symbols_1 */ "1",
        /* keyspec_symbols_2 */ "2",
        /* keyspec_symbols_3 */ "3",
        /* keyspec_symbols_4 */ "4",
        /* keyspec_symbols_5 */ "5",
        /* keyspec_symbols_6 */ "6",
        /* keyspec_symbols_7 */ "7",
        /* keyspec_symbols_8 */ "8",
        /* keyspec_symbols_9 */ "9",
        /* keyspec_symbols_0 */ "0",
        /* keyspec_currency */ "\u20B9",
        /* morekeys_period */ "!text/morekeys_punctuation",
        // Comma key
        /* keyspec_comma */ ",",
        // Period key
        /* keyspec_period */ ".",
        /* keyspec_tablet_period */ ".",
        /* morekeys_tablet_period */ "!text/morekeys_tablet_punctuation",
        /* morekeys_a ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ morekeys_w */
        // $ - I'm making ₹ as default
        // U+00A2: "¢" CENT SIGN
        // U+00A3: "£" POUND SIGN
        // U+20AC: "€" EURO SIGN
        // U+00A5: "¥" YEN SIGN
        // U+20B1: "₱" PESO SIGN
        /* morekeys_currency_dollar */ "$,\u00A2,\u00A3,\u20AC,\u00A5,\u20B1",
        /* keyspec_symbols_semicolon */ ";",
        /* label_go_key */ "!string/label_go_key",
        /* label_send_key */ "!string/label_send_key",
        /* label_next_key */ "!string/label_next_key",
        /* label_done_key */ "!string/label_done_key",
        /* label_search_key */ "!string/label_search_key",
        /* label_previous_key */ "!string/label_previous_key",
        /* label_pause_key */ "!string/label_pause_key",
        /* label_wait_key */ "!string/label_wait_key",
        /* morekeys_v ~ */
        EMPTY, EMPTY, EMPTY,
        /* ~ morekeys_x */
        /* keyspec_q */ "q",
        /* keyspec_w */ "w",
        /* keyspec_y */ "y",
        /* keyspec_x */ "x",
        /* keyspec_nordic_row1_11 ~ */
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY, EMPTY,
        EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ morekeys_swiss_row2_11 */
        /* single_quotes */ "!text/single_lqm_rqm",
        /* double_quotes */ "!text/double_lqm_rqm",
        /* single_angle_quotes */ "!text/single_laqm_raqm",
        /* double_angle_quotes */ "!text/double_laqm_raqm",
        /* morekeys_currency_generic */ "$,\u00A2,\u20AC,\u00A3,\u00A5,\u20B1",
        /* morekeys_punctuation */ "!autoColumnOrder!8,\\,,?,!,#,!text/keyspec_right_parenthesis,!text/keyspec_left_parenthesis,/,;,',@,:,-,\",+,\\%,&",
        /* morekeys_tablet_punctuation */ "!autoColumnOrder!7,\\,,',#,!text/keyspec_right_parenthesis,!text/keyspec_left_parenthesis,/,;,@,:,-,\",+,\\%,&",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        /* keyspec_spanish_row2_10 */ "\u00F1",
        // U+2020: "†" DAGGER
        // U+2021: "‡" DOUBLE DAGGER
        // U+2605: "★" BLACK STAR
        /* morekeys_star */ "\u2020,\u2021,\u2605",
        // U+266A: "♪" EIGHTH NOTE
        // U+2665: "♥" BLACK HEART SUIT
        // U+2660: "♠" BLACK SPADE SUIT
        // U+2666: "♦" BLACK DIAMOND SUIT
        // U+2663: "♣" BLACK CLUB SUIT
        /* morekeys_bullet */ "\u266A,\u2665,\u2660,\u2666,\u2663",
        // U+00B1: "±" PLUS-MINUS SIGN
        /* morekeys_plus */ "\u00B1",
        /* morekeys_left_parenthesis */ "!fixedColumnOrder!3,!text/keyspecs_left_parenthesis_more_keys",
        /* morekeys_right_parenthesis */ "!fixedColumnOrder!3,!text/keyspecs_right_parenthesis_more_keys",
        /* morekeys_less_than */ "!fixedColumnOrder!3,!text/keyspec_left_single_angle_quote,!text/keyspec_less_than_equal,!text/keyspec_left_double_angle_quote",
        /* morekeys_greater_than */ "!fixedColumnOrder!3,!text/keyspec_right_single_angle_quote,!text/keyspec_greater_than_equal,!text/keyspec_right_double_angle_quote",
        /* morekeys_arabic_diacritics */ EMPTY,
        // U+00B9: "¹" SUPERSCRIPT ONE
        // U+00BD: "½" VULGAR FRACTION ONE HALF
        // U+2153: "⅓" VULGAR FRACTION ONE THIRD
        // U+00BC: "¼" VULGAR FRACTION ONE QUARTER
        // U+215B: "⅛" VULGAR FRACTION ONE EIGHTH
        /* morekeys_symbols_1 */ "\u00B9,\u00BD,\u2153,\u00BC,\u215B",
        // U+00B2: "²" SUPERSCRIPT TWO
        // U+2154: "⅔" VULGAR FRACTION TWO THIRDS
        /* morekeys_symbols_2 */ "\u00B2,\u2154",
        // U+00B3: "³" SUPERSCRIPT THREE
        // U+00BE: "¾" VULGAR FRACTION THREE QUARTERS
        // U+215C: "⅜" VULGAR FRACTION THREE EIGHTHS
        /* morekeys_symbols_3 */ "\u00B3,\u00BE,\u215C",
        // U+2074: "⁴" SUPERSCRIPT FOUR
        /* morekeys_symbols_4 */ "\u2074",
        // U+215D: "⅝" VULGAR FRACTION FIVE EIGHTHS
        /* morekeys_symbols_5 */ "\u215D",
        /* morekeys_symbols_6 */ EMPTY,
        // U+215E: "⅞" VULGAR FRACTION SEVEN EIGHTHS
        /* morekeys_symbols_7 */ "\u215E",
        /* morekeys_symbols_8 */ EMPTY,
        /* morekeys_symbols_9 */ EMPTY,
        // U+207F: "ⁿ" SUPERSCRIPT LATIN SMALL LETTER N
        // U+2205: "∅" EMPTY SET
        /* morekeys_symbols_0 */ "\u207F,\u2205",
        // The all letters need to be mirrored are found at
        // http://www.unicode.org/Public/6.1.0/ucd/BidiMirroring.txt
        // U+2039: "‹" SINGLE LEFT-POINTING ANGLE QUOTATION MARK
        // U+203A: "›" SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
        // U+2264: "≤" LESS-THAN OR EQUAL TO
        // U+2265: "≥" GREATER-THAN EQUAL TO
        // U+00AB: "«" LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
        // U+00BB: "»" RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
        /* keyspec_left_parenthesis */ "(",
        /* keyspec_right_parenthesis */ ")",
        /* keyspec_left_square_bracket */ "[",
        /* keyspec_right_square_bracket */ "]",
        /* keyspec_left_curly_bracket */ "{",
        /* keyspec_right_curly_bracket */ "}",
        /* keyspec_less_than */ "<",
        /* keyspec_greater_than */ ">",
        /* keyspec_less_than_equal */ "\u2264",
        /* keyspec_greater_than_equal */ "\u2265",
        /* keyspec_left_double_angle_quote */ "\u00AB",
        /* keyspec_right_double_angle_quote */ "\u00BB",
        /* keyspec_left_single_angle_quote */ "\u2039",
        /* keyspec_right_single_angle_quote */ "\u203A",
        /* keyspec_tablet_comma */ ",",
        /* keyhintlabel_tablet_comma ~ */
        EMPTY, EMPTY, EMPTY, EMPTY,
        /* ~ keyhintlabel_tablet_period */
        /* keyspec_symbols_question */ "?",
        /* keyspec_symbols_percent */ "%",
        // U+00A1: "¡" INVERTED EXCLAMATION MARK
        /* morekeys_exclamation */ "\u00A1",
        // U+00BF: "¿" INVERTED QUESTION MARK
        /* morekeys_question */ "\u00BF",
        /* morekeys_symbols_semicolon */ EMPTY,
        // U+2030: "‰" PER MILLE SIGN
        /* morekeys_symbols_percent */ "\u2030",
        /* morekeys_am_pm */ "!fixedColumnOrder!2,!hasLabels!,!text/keylabel_time_am,!text/keylabel_time_pm",
        /* keyspec_settings */ "!icon/settings_key|!code/key_settings",
        /* keyspec_shortcut */ "!icon/shortcut_key|!code/key_shortcut",
        /* keyspec_action_next */ "!hasLabels!,!text/label_next_key|!code/key_action_next",
        /* keyspec_action_previous */ "!hasLabels!,!text/label_previous_key|!code/key_action_previous",
        // Label for "switch to more symbol" modifier key ("= \ <"). Must be short to fit on key!
        /* keylabel_to_more_symbol */ "= \\\\ <",
        // Label for "switch to more symbol" modifier key on tablets.  Must be short to fit on key!
        /* keylabel_tablet_to_more_symbol */ "~ [ <",
        // Label for "switch to phone numeric" key.  Must be short to fit on key!
        /* keylabel_to_phone_numeric */ "123",
        // Label for "switch to phone symbols" key.  Must be short to fit on key!
        // U+FF0A: "＊" FULLWIDTH ASTERISK
        // U+FF03: "＃" FULLWIDTH NUMBER SIGN
        /* keylabel_to_phone_symbols */ "\uFF0A\uFF03",
        // Key label for "ante meridiem"
        /* keylabel_time_am */ "AM",
        // Key label for "post meridiem"
        /* keylabel_time_pm */ "PM",
        /* keyspec_popular_domain */ ".com",
        // popular web domains for the locale - most popular, displayed on the keyboard
        /* morekeys_popular_domain */ "!hasLabels!,.net,.org,.gov,.edu",
        /* keyspecs_left_parenthesis_more_keys */ "!text/keyspec_less_than,!text/keyspec_left_curly_bracket,!text/keyspec_left_square_bracket",
        /* keyspecs_right_parenthesis_more_keys */ "!text/keyspec_greater_than,!text/keyspec_right_curly_bracket,!text/keyspec_right_square_bracket",
        // The following characters don't need BIDI mirroring.
        // U+2018: "‘" LEFT SINGLE QUOTATION MARK
        // U+2019: "’" RIGHT SINGLE QUOTATION MARK
        // U+201A: "‚" SINGLE LOW-9 QUOTATION MARK
        // U+201C: "“" LEFT DOUBLE QUOTATION MARK
        // U+201D: "”" RIGHT DOUBLE QUOTATION MARK
        // U+201E: "„" DOUBLE LOW-9 QUOTATION MARK
        // Abbreviations are:
        // laqm: LEFT-POINTING ANGLE QUOTATION MARK
        // raqm: RIGHT-POINTING ANGLE QUOTATION MARK
        // lqm: LEFT QUOTATION MARK
        // rqm: RIGHT QUOTATION MARK
        // 9qm: LOW-9 QUOTATION MARK
        // The following each quotation mark pair consist of
        // <opening quotation mark>, <closing quotation mark>
        // and is named after (single|double)_<opening quotation mark>_<closing quotation mark>.
        /* single_laqm_raqm */ "!text/keyspec_left_single_angle_quote,!text/keyspec_right_single_angle_quote",
        /* single_raqm_laqm */ "!text/keyspec_right_single_angle_quote,!text/keyspec_left_single_angle_quote",
        /* double_laqm_raqm */ "!text/keyspec_left_double_angle_quote,!text/keyspec_right_double_angle_quote",
        /* double_raqm_laqm */ "!text/keyspec_right_double_angle_quote,!text/keyspec_left_double_angle_quote",
        // The following each quotation mark triplet consists of
        // <another quotation mark>, <opening quotation mark>, <closing quotation mark>
        // and is named after (single|double)_<opening quotation mark>_<closing quotation mark>.
        /* single_lqm_rqm */ "\u201A,\u2018,\u2019",
        /* single_9qm_lqm */ "\u2019,\u201A,\u2018",
        /* single_9qm_rqm */ "\u2018,\u201A,\u2019",
        /* single_rqm_9qm */ "\u2018,\u2019,\u201A",
        /* double_lqm_rqm */ "\u201E,\u201C,\u201D",
        /* double_9qm_lqm */ "\u201D,\u201E,\u201C",
        /* double_9qm_rqm */ "\u201C,\u201E,\u201D",
        /* double_rqm_9qm */ "\u201C,\u201D,\u201E",
        /* morekeys_single_quote */ "!fixedColumnOrder!5,!text/single_quotes,!text/single_angle_quotes",
        /* morekeys_double_quote */ "!fixedColumnOrder!5,!text/double_quotes,!text/double_angle_quotes",
        /* morekeys_tablet_double_quote */ "!fixedColumnOrder!6,!text/double_quotes,!text/single_quotes,!text/double_angle_quotes,!text/single_angle_quotes",
        /* keyspec_emoji_action_key */ "!icon/emoji_action_key|!code/key_emoji",
    };

    /* Locale ar: Arabic */
    private static final String[] TEXTS_ar = {
        /* keylabel_to_alpha ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null,
        /* ~ keyspec_currency */
        /* morekeys_period */ "!autoColumnOrder!8,\u064D,\u064B,\u064C,\u0652,\u0651,\u0640,\u061F,\u061B,\u0650,\u064E,\u064F,\u0653,\u0670,\u0656,\u0654,\u0655",
        /* keyspec_comma */ "\u060C",
    };

    /* Locale as_IN: Assamese (India) */
    private static final String[] TEXTS_as_IN = {
        /* keylabel_to_alpha */ "\u0985",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
        /* keylabel_to_symbol */ "\u09E7\u09E8\u09E9",
        /* keyspec_symbols_1 */ "\u09E7",
        /* keyspec_symbols_2 */ "\u09E8",
        /* keyspec_symbols_3 */ "\u09E9",
        /* keyspec_symbols_4 */ "\u09EA",
        /* keyspec_symbols_5 */ "\u09EB",
        /* keyspec_symbols_6 */ "\u09EC",
        /* keyspec_symbols_7 */ "\u09ED",
        /* keyspec_symbols_8 */ "\u09EE",
        /* keyspec_symbols_9 */ "\u09EF",
        /* keyspec_symbols_0 */ "\u09E6",
        /* keyspec_currency */ null,
        /* morekeys_period */ null,
        /* keyspec_comma */ "\u09CD",
        /* keyspec_period ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null,
        /* ~ morekeys_w */
        /* morekeys_currency_dollar */ "\u09F3",
        /* keyspec_symbols_semicolon */ "\u0983",
    };

    /* Locale bn_IN: Bangla (India) */
    private static final String[] TEXTS_bn_IN = {
        /* keylabel_to_alpha */ "\u0985",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
        /* keylabel_to_symbol */ "\u09E7\u09E8\u09E9",
        /* keyspec_symbols_1 */ "\u09E7",
        /* keyspec_symbols_2 */ "\u09E8",
        /* keyspec_symbols_3 */ "\u09E9",
        /* keyspec_symbols_4 */ "\u09EA",
        /* keyspec_symbols_5 */ "\u09EB",
        /* keyspec_symbols_6 */ "\u09EC",
        /* keyspec_symbols_7 */ "\u09ED",
        /* keyspec_symbols_8 */ "\u09EE",
        /* keyspec_symbols_9 */ "\u09EF",
        /* keyspec_symbols_0 */ "\u09E6",
    };

    /* Locale en: English */
    private static final String[] TEXTS_en = {
        /* keylabel_to_alpha ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_tablet_period */
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        /* morekeys_a */ "\u00E0,\u00E1,\u00E2,\u00E4,\u00E6,\u00E3,\u00E5,\u0101",
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        /* morekeys_e */ "\u00E9,\u00E8,\u00EA,\u00EB,\u0113",
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        /* morekeys_i */ "\u00ED,\u00EE,\u00EF,\u012B,\u00EC",
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        /* morekeys_o */ "\u00F3,\u00F4,\u00F6,\u00F2,\u0153,\u00F8,\u014D,\u00F5",
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        /* morekeys_u */ "\u00FA,\u00FB,\u00FC,\u00F9,\u016B",
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        /* morekeys_s */ "\u00DF",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        /* morekeys_n */ "\u00F1",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        /* morekeys_c */ "\u00E7",
    };

    /* Locale gu_IN: Gujarati (India) */
    private static final String[] TEXTS_gu_IN = {
        // Label for "switch to alphabetic" key.
        // U+0A95: "ક" GUJARATI LETTER KA
        // U+0A96: "ખ" GUJARATI LETTER KHA
        // U+0A97: "ગ" GUJARATI LETTER GA
        /* keylabel_to_alpha */ "\u0A95\u0A96\u0A97",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
        // Label for "switch to symbols" key.
        /* keylabel_to_symbol */ "?\u0AE7\u0AE8\u0AE9",
        // U+0AE7: "૧" GUJARATI DIGIT ONE
        /* keyspec_symbols_1 */ "\u0AE7",
        // U+0AE8: "૨" GUJARATI DIGIT TWO
        /* keyspec_symbols_2 */ "\u0AE8",
        // U+0AE9: "૩" GUJARATI DIGIT THREE
        /* keyspec_symbols_3 */ "\u0AE9",
        // U+0AEA: "૪" GUJARATI DIGIT FOUR
        /* keyspec_symbols_4 */ "\u0AEA",
        // U+0AEB: "૫" GUJARATI DIGIT FIVE
        /* keyspec_symbols_5 */ "\u0AEB",
        // U+0AEC: "૬" GUJARATI DIGIT SIX
        /* keyspec_symbols_6 */ "\u0AEC",
        // U+0AED: "૭" GUJARATI DIGIT SEVEN
        /* keyspec_symbols_7 */ "\u0AED",
        // U+0AEE: "૮" GUJARATI DIGIT EIGHT
        /* keyspec_symbols_8 */ "\u0AEE",
        // U+0AEF: "૯" GUJARATI DIGIT NINE
        /* keyspec_symbols_9 */ "\u0AEF",
        // U+0AE6: "૦" GUJARATI DIGIT ZERO
        /* keyspec_symbols_0 */ "\u0AE6",
        // U+20B9: "₹" INDIAN RUPEE SIGN
        /* keyspec_currency */ "\u20B9",
        /* morekeys_period */ "!autoColumnOrder!9,\\,,.,?,!,#,),(,/,;,',@,:,-,\",+,\\%,&",
        /* keyspec_comma */ null,
        // U+0964: "." GUJARATI .
        /* keyspec_period */ "\u0ABC",
        /* keyspec_tablet_period */ "\u0ABC",
        /* morekeys_tablet_period */ "!autoColumnOrder!8,\\,,.,',#,),(,/,;,@,:,-,\",+,\\%,&",
    };

    /* Locale hi: Hindi */
    private static final String[] TEXTS_hi = {
        // Label for "switch to alphabetic" key.
        // U+0915: "क" DEVANAGARI LETTER KA
        // U+0916: "ख" DEVANAGARI LETTER KHA
        // U+0917: "ग" DEVANAGARI LETTER GA
        /* keylabel_to_alpha */ "\u0915\u0916\u0917",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
        // Label for "switch to symbols" key.
        /* keylabel_to_symbol */ "?\u0967\u0968\u0969",
        // U+0967: "१" DEVANAGARI DIGIT ONE
        /* keyspec_symbols_1 */ "\u0967",
        // U+0968: "२" DEVANAGARI DIGIT TWO
        /* keyspec_symbols_2 */ "\u0968",
        // U+0969: "३" DEVANAGARI DIGIT THREE
        /* keyspec_symbols_3 */ "\u0969",
        // U+096A: "४" DEVANAGARI DIGIT FOUR
        /* keyspec_symbols_4 */ "\u096A",
        // U+096B: "५" DEVANAGARI DIGIT FIVE
        /* keyspec_symbols_5 */ "\u096B",
        // U+096C: "६" DEVANAGARI DIGIT SIX
        /* keyspec_symbols_6 */ "\u096C",
        // U+096D: "७" DEVANAGARI DIGIT SEVEN
        /* keyspec_symbols_7 */ "\u096D",
        // U+096E: "८" DEVANAGARI DIGIT EIGHT
        /* keyspec_symbols_8 */ "\u096E",
        // U+096F: "९" DEVANAGARI DIGIT NINE
        /* keyspec_symbols_9 */ "\u096F",
        // U+0966: "०" DEVANAGARI DIGIT ZERO
        /* keyspec_symbols_0 */ "\u0966",
        // U+20B9: "₹" INDIAN RUPEE SIGN
        /* keyspec_currency */ "\u20B9",
        /* morekeys_period */ "!autoColumnOrder!9,\\,,.,?,!,#,),(,/,;,',@,:,-,\",+,\\%,&",
        /* keyspec_comma */ null,
        // U+0964: "।" DEVANAGARI DANDA
        /* keyspec_period */ "\u0964",
        /* keyspec_tablet_period */ "\u0964",
        /* morekeys_tablet_period */ "!autoColumnOrder!8,\\,,.,',#,),(,/,;,@,:,-,\",+,\\%,&",
    };

    /* Locale hi_ZZ: Hindi (Unknown Region) */
    private static final String[] TEXTS_hi_ZZ = {
        /* keylabel_to_alpha ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null,
        /* ~ keyspec_symbols_0 */
        // U+20B9: "₹" INDIAN RUPEE SIGN
        /* keyspec_currency */ "\u20B9",
        /* morekeys_period ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null,
        /* ~ keyspec_symbols_semicolon */
        /* label_go_key */ "Go",
        /* label_send_key */ "Send",
        /* label_next_key */ "Next",
        /* label_done_key */ "Done",
        /* label_search_key */ "Search",
        /* label_previous_key */ "Prev",
        /* label_pause_key */ "Pause",
        /* label_wait_key */ "Wait",
    };

    /* Locale kn: Kannada */
    private static final String[] TEXTS_kn = {
        /* keylabel_to_alpha */ "\u0C85",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
        /* keylabel_to_symbol */ "\u0CE7\u0CE8\u0CE9",
        /* keyspec_symbols_1 */ "\u0CE7",
        /* keyspec_symbols_2 */ "\u0CE8",
        /* keyspec_symbols_3 */ "\u0CE9",
        /* keyspec_symbols_4 */ "\u0CEA",
        /* keyspec_symbols_5 */ "\u0CEB",
        /* keyspec_symbols_6 */ "\u0CEC",
        /* keyspec_symbols_7 */ "\u0CED",
        /* keyspec_symbols_8 */ "\u0CEE",
        /* keyspec_symbols_9 */ "\u0CEF",
        /* keyspec_symbols_0 */ "\u0CE6",
    };

    /* Locale ml: Malayalam */
    private static final String[] TEXTS_ml = {
        /* keylabel_to_alpha */ "\u0D05",
        // U+0966: "൧" MALAYALAM DIGIT ONE
        /* additional_morekeys_symbols_1 */ "\u0D67",
        // U+0966: "൨" MALAYALAM DIGIT TWO
        /* additional_morekeys_symbols_2 */ "\u0D68",
        // U+0966: "൩" MALAYALAM DIGIT THREE
        /* additional_morekeys_symbols_3 */ "\u0D69",
        // U+0966: "൪" MALAYALAM DIGIT FOUR
        /* additional_morekeys_symbols_4 */ "\u0D6A",
        // U+0966: "൫" MALAYALAM DIGIT FIVE
        /* additional_morekeys_symbols_5 */ "\u0D6B",
        // U+0966: "൬" MALAYALAM DIGIT SIX
        /* additional_morekeys_symbols_6 */ "\u0D6C",
        // U+0966: "൭" MALAYALAM DIGIT SEVEN
        /* additional_morekeys_symbols_7 */ "\u0D6D",
        // U+0966: "൮" MALAYALAM DIGIT EIGHT
        /* additional_morekeys_symbols_8 */ "\u0D6E",
        // U+0966: "൯" MALAYALAM DIGIT NINE
        /* additional_morekeys_symbols_9 */ "\u0D6F",
        // U+0966: "൦" MALAYALAM DIGIT ZERO
        /* additional_morekeys_symbols_0 */ "\u0D66",
    };

    /* Locale mr: Marathi */
    private static final String[] TEXTS_mr = {
        // Label for "switch to alphabetic" key.
        // U+0915: "क" DEVANAGARI LETTER KA
        // U+0916: "ख" DEVANAGARI LETTER KHA
        // U+0917: "ग" DEVANAGARI LETTER GA
        /* keylabel_to_alpha */ "\u0915\u0916\u0917",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
        // Label for "switch to symbols" key.
        /* keylabel_to_symbol */ "?\u0967\u0968\u0969",
        // U+0967: "१" DEVANAGARI DIGIT ONE
        /* keyspec_symbols_1 */ "\u0967",
        // U+0968: "२" DEVANAGARI DIGIT TWO
        /* keyspec_symbols_2 */ "\u0968",
        // U+0969: "३" DEVANAGARI DIGIT THREE
        /* keyspec_symbols_3 */ "\u0969",
        // U+096A: "४" DEVANAGARI DIGIT FOUR
        /* keyspec_symbols_4 */ "\u096A",
        // U+096B: "५" DEVANAGARI DIGIT FIVE
        /* keyspec_symbols_5 */ "\u096B",
        // U+096C: "६" DEVANAGARI DIGIT SIX
        /* keyspec_symbols_6 */ "\u096C",
        // U+096D: "७" DEVANAGARI DIGIT SEVEN
        /* keyspec_symbols_7 */ "\u096D",
        // U+096E: "८" DEVANAGARI DIGIT EIGHT
        /* keyspec_symbols_8 */ "\u096E",
        // U+096F: "९" DEVANAGARI DIGIT NINE
        /* keyspec_symbols_9 */ "\u096F",
        // U+0966: "०" DEVANAGARI DIGIT ZERO
        /* keyspec_symbols_0 */ "\u0966",
        // U+20B9: "₹" INDIAN RUPEE SIGN
        /* keyspec_currency */ "\u20B9",
    };

    /* Locale my: Burmese */
    private static final String[] TEXTS_my = {
        /* keylabel_to_alpha */ "\u1000",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
        // Label for "switch to symbols" key.
        /* keylabel_to_symbol */ "\u1041",
        /* keyspec_symbols_1 */ "\u1041",
        /* keyspec_symbols_2 */ "\u1042",
        /* keyspec_symbols_3 */ "\u1043",
        /* keyspec_symbols_4 */ "\u1044",
        /* keyspec_symbols_5 */ "\u1045",
        /* keyspec_symbols_6 */ "\u1046",
        /* keyspec_symbols_7 */ "\u1047",
        /* keyspec_symbols_8 */ "\u1048",
        /* keyspec_symbols_9 */ "\u1049",
        /* keyspec_symbols_0 */ "\u1040",
    };

    /* Locale ne: Nepali */
    private static final String[] TEXTS_ne = {
        // Label for "switch to alphabetic" key.
        // U+0915: "क" DEVANAGARI LETTER KA
        // U+0916: "ख" DEVANAGARI LETTER KHA
        // U+0917: "ग" DEVANAGARI LETTER GA
        /* keylabel_to_alpha */ "\u0915\u0916\u0917",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
        // Label for "switch to symbols" key.
        /* keylabel_to_symbol */ "?\u0967\u0968\u0969",
        // U+0967: "१" DEVANAGARI DIGIT ONE
        /* keyspec_symbols_1 */ "\u0967",
        // U+0968: "२" DEVANAGARI DIGIT TWO
        /* keyspec_symbols_2 */ "\u0968",
        // U+0969: "३" DEVANAGARI DIGIT THREE
        /* keyspec_symbols_3 */ "\u0969",
        // U+096A: "४" DEVANAGARI DIGIT FOUR
        /* keyspec_symbols_4 */ "\u096A",
        // U+096B: "५" DEVANAGARI DIGIT FIVE
        /* keyspec_symbols_5 */ "\u096B",
        // U+096C: "६" DEVANAGARI DIGIT SIX
        /* keyspec_symbols_6 */ "\u096C",
        // U+096D: "७" DEVANAGARI DIGIT SEVEN
        /* keyspec_symbols_7 */ "\u096D",
        // U+096E: "८" DEVANAGARI DIGIT EIGHT
        /* keyspec_symbols_8 */ "\u096E",
        // U+096F: "९" DEVANAGARI DIGIT NINE
        /* keyspec_symbols_9 */ "\u096F",
        // U+0966: "०" DEVANAGARI DIGIT ZERO
        /* keyspec_symbols_0 */ "\u0966",
        // U+0930/U+0941/U+002E "रु." NEPALESE RUPEE SIGN
        /* keyspec_currency */ "\u0930\u0941.",
        /* morekeys_period */ "!autoColumnOrder!9,.,\\,,?,!,#,),(,/,;,',@,:,-,\",+,\\%,&",
        /* keyspec_comma */ null,
        // U+0964: "।" DEVANAGARI DANDA
        /* keyspec_period */ "\u0964",
        /* keyspec_tablet_period */ "\u0964",
        /* morekeys_tablet_period */ "!autoColumnOrder!8,.,\\,,',#,),(,/,;,@,:,-,\",+,\\%,&",
    };

    /* Locale or_IN: Odia (India) */
    private static final String[] TEXTS_or_IN = {
        /* keylabel_to_alpha */ "\u0B05",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
        /* keylabel_to_symbol */ "\u0B67\u0B68\u0B69",
        /* keyspec_symbols_1 */ "\u0B67",
        /* keyspec_symbols_2 */ "\u0B68",
        /* keyspec_symbols_3 */ "\u0B69",
        /* keyspec_symbols_4 */ "\u0B6A",
        /* keyspec_symbols_5 */ "\u0B6B",
        /* keyspec_symbols_6 */ "\u0B6C",
        /* keyspec_symbols_7 */ "\u0B6D",
        /* keyspec_symbols_8 */ "\u0B6E",
        /* keyspec_symbols_9 */ "\u0B6F",
        /* keyspec_symbols_0 */ "\u0B66",
        /* keyspec_currency */ null,
        /* morekeys_period */ null,
        /* keyspec_comma */ "\u0B3C",
    };

    /* Locale pa_IN: Punjabi (India) */
    private static final String[] TEXTS_pa_IN = {
        /* keylabel_to_alpha */ "\u0A05",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
        /* keylabel_to_symbol */ "\u0A67\u0A68\u0A69",
        /* keyspec_symbols_1 */ "\u0A67",
        /* keyspec_symbols_2 */ "\u0A68",
        /* keyspec_symbols_3 */ "\u0A69",
        /* keyspec_symbols_4 */ "\u0A6A",
        /* keyspec_symbols_5 */ "\u0A6B",
        /* keyspec_symbols_6 */ "\u0A6C",
        /* keyspec_symbols_7 */ "\u0A6D",
        /* keyspec_symbols_8 */ "\u0A6E",
        /* keyspec_symbols_9 */ "\u0A6F",
        /* keyspec_symbols_0 */ "\u0A66",
    };

    /* Locale ta_IN: Tamil (India) */
    private static final String[] TEXTS_ta_IN = {
        /* keylabel_to_alpha */ "\u0B85",
        // U+0BE7: "௧" TAMIL DIGIT ONE
        /* additional_morekeys_symbols_1 */ "\u0BE7",
        // U+0BE8: "௨" TAMIL DIGIT TWO
        /* additional_morekeys_symbols_2 */ "\u0BE8",
        // U+0BE9: "௩" TAMIL DIGIT THREE
        /* additional_morekeys_symbols_3 */ "\u0BE9",
        // U+0BEA: "௪" TAMIL DIGIT FOUR
        /* additional_morekeys_symbols_4 */ "\u0BEA",
        // U+0BEB: "௫" TAMIL DIGIT FIVE
        /* additional_morekeys_symbols_5 */ "\u0BEB",
        // U+0BEC: "௬" TAMIL DIGIT SIX
        /* additional_morekeys_symbols_6 */ "\u0BEC",
        // U+0BED: "௭" TAMIL DIGIT SEVEN
        /* additional_morekeys_symbols_7 */ "\u0BED",
        // U+0BEE: "௮" TAMIL DIGIT EIGHT
        /* additional_morekeys_symbols_8 */ "\u0BEE",
        // U+0BEF: "௯" TAMIL DIGIT NINE
        /* additional_morekeys_symbols_9 */ "\u0BEF",
        // U+0BE6: "௦" TAMIL DIGIT ZERO
        /* additional_morekeys_symbols_0 */ "\u0BE6",
        /* keylabel_to_symbol */ "\u0BE7\u0BE8\u0BE9",
    };

    /* Locale ta_LK: Tamil (Sri Lanka) */
    private static final String[] TEXTS_ta_LK = {
        // Label for "switch to alphabetic" key.
        // U+0BA4: "த" TAMIL LETTER TA
        // U+0BAE/U+0BBF: "மி" TAMIL LETTER MA/TAMIL VOWEL SIGN I
        // U+0BB4/U+0BCD: "ழ்" TAMIL LETTER LLLA/TAMIL SIGN VIRAMA
        /* keylabel_to_alpha */ "\u0BA4\u0BAE\u0BBF\u0BB4\u0BCD",
        /* additional_morekeys_symbols_1 ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null,
        /* ~ keyspec_symbols_0 */
        // U+0DBB/U+0DD4: "රු" SINHALA LETTER RAYANNA/SINHALA VOWEL SIGN KETTI PAA-PILLA
        /* keyspec_currency */ "\u0DBB\u0DD4",
    };

    /* Locale ta_SG: Tamil (Singapore) */
    private static final String[] TEXTS_ta_SG = {
        // Label for "switch to alphabetic" key.
        // U+0BA4: "த" TAMIL LETTER TA
        // U+0BAE/U+0BBF: "மி" TAMIL LETTER MA/TAMIL VOWEL SIGN I
        // U+0BB4/U+0BCD: "ழ்" TAMIL LETTER LLLA/TAMIL SIGN VIRAMA
        /* keylabel_to_alpha */ "\u0BA4\u0BAE\u0BBF\u0BB4\u0BCD",
    };

    /* Locale te: Telugu */
    private static final String[] TEXTS_te = {
        /* keylabel_to_alpha */ "\u0C05",
        /* additional_morekeys_symbols_1 */ "1",
        /* additional_morekeys_symbols_2 */ "2",
        /* additional_morekeys_symbols_3 */ "3",
        /* additional_morekeys_symbols_4 */ "4",
        /* additional_morekeys_symbols_5 */ "5",
        /* additional_morekeys_symbols_6 */ "6",
        /* additional_morekeys_symbols_7 */ "7",
        /* additional_morekeys_symbols_8 */ "8",
        /* additional_morekeys_symbols_9 */ "9",
        /* additional_morekeys_symbols_0 */ "0",
        /* keylabel_to_symbol */ "\u0C67\u0C68\u0C69",
        /* keyspec_symbols_1 */ "\u0C67",
        /* keyspec_symbols_2 */ "\u0C68",
        /* keyspec_symbols_3 */ "\u0C69",
        /* keyspec_symbols_4 */ "\u0C6A",
        /* keyspec_symbols_5 */ "\u0C6B",
        /* keyspec_symbols_6 */ "\u0C6C",
        /* keyspec_symbols_7 */ "\u0C6D",
        /* keyspec_symbols_8 */ "\u0C6E",
        /* keyspec_symbols_9 */ "\u0C6F",
        /* keyspec_symbols_0 */ "\u0C66",
    };

    /* Locale zz: Alphabet */
    private static final String[] TEXTS_zz = {
        /* keylabel_to_alpha ~ */
        null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
        null, null, null, null, null, null, null, null, null, null, null, null, null,
        /* ~ morekeys_tablet_period */
        // U+00E0: "à" LATIN SMALL LETTER A WITH GRAVE
        // U+00E1: "á" LATIN SMALL LETTER A WITH ACUTE
        // U+00E2: "â" LATIN SMALL LETTER A WITH CIRCUMFLEX
        // U+00E3: "ã" LATIN SMALL LETTER A WITH TILDE
        // U+00E4: "ä" LATIN SMALL LETTER A WITH DIAERESIS
        // U+00E5: "å" LATIN SMALL LETTER A WITH RING ABOVE
        // U+00E6: "æ" LATIN SMALL LETTER AE
        // U+0101: "ā" LATIN SMALL LETTER A WITH MACRON
        // U+0103: "ă" LATIN SMALL LETTER A WITH BREVE
        // U+0105: "ą" LATIN SMALL LETTER A WITH OGONEK
        // U+00AA: "ª" FEMININE ORDINAL INDICATOR
        /* morekeys_a */ "\u00E0,\u00E1,\u00E2,\u00E3,\u00E4,\u00E5,\u00E6,\u0101,\u0103,\u0105,\u00AA",
        // U+00E8: "è" LATIN SMALL LETTER E WITH GRAVE
        // U+00E9: "é" LATIN SMALL LETTER E WITH ACUTE
        // U+00EA: "ê" LATIN SMALL LETTER E WITH CIRCUMFLEX
        // U+00EB: "ë" LATIN SMALL LETTER E WITH DIAERESIS
        // U+0113: "ē" LATIN SMALL LETTER E WITH MACRON
        // U+0115: "ĕ" LATIN SMALL LETTER E WITH BREVE
        // U+0117: "ė" LATIN SMALL LETTER E WITH DOT ABOVE
        // U+0119: "ę" LATIN SMALL LETTER E WITH OGONEK
        // U+011B: "ě" LATIN SMALL LETTER E WITH CARON
        /* morekeys_e */ "\u00E8,\u00E9,\u00EA,\u00EB,\u0113,\u0115,\u0117,\u0119,\u011B",
        // U+00EC: "ì" LATIN SMALL LETTER I WITH GRAVE
        // U+00ED: "í" LATIN SMALL LETTER I WITH ACUTE
        // U+00EE: "î" LATIN SMALL LETTER I WITH CIRCUMFLEX
        // U+00EF: "ï" LATIN SMALL LETTER I WITH DIAERESIS
        // U+0129: "ĩ" LATIN SMALL LETTER I WITH TILDE
        // U+012B: "ī" LATIN SMALL LETTER I WITH MACRON
        // U+012D: "ĭ" LATIN SMALL LETTER I WITH BREVE
        // U+012F: "į" LATIN SMALL LETTER I WITH OGONEK
        // U+0131: "ı" LATIN SMALL LETTER DOTLESS I
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* morekeys_i */ "\u00EC,\u00ED,\u00EE,\u00EF,\u0129,\u012B,\u012D,\u012F,\u0131,\u0133",
        // U+00F2: "ò" LATIN SMALL LETTER O WITH GRAVE
        // U+00F3: "ó" LATIN SMALL LETTER O WITH ACUTE
        // U+00F4: "ô" LATIN SMALL LETTER O WITH CIRCUMFLEX
        // U+00F5: "õ" LATIN SMALL LETTER O WITH TILDE
        // U+00F6: "ö" LATIN SMALL LETTER O WITH DIAERESIS
        // U+00F8: "ø" LATIN SMALL LETTER O WITH STROKE
        // U+014D: "ō" LATIN SMALL LETTER O WITH MACRON
        // U+014F: "ŏ" LATIN SMALL LETTER O WITH BREVE
        // U+0151: "ő" LATIN SMALL LETTER O WITH DOUBLE ACUTE
        // U+0153: "œ" LATIN SMALL LIGATURE OE
        // U+00BA: "º" MASCULINE ORDINAL INDICATOR
        /* morekeys_o */ "\u00F2,\u00F3,\u00F4,\u00F5,\u00F6,\u00F8,\u014D,\u014F,\u0151,\u0153,\u00BA",
        // U+00F9: "ù" LATIN SMALL LETTER U WITH GRAVE
        // U+00FA: "ú" LATIN SMALL LETTER U WITH ACUTE
        // U+00FB: "û" LATIN SMALL LETTER U WITH CIRCUMFLEX
        // U+00FC: "ü" LATIN SMALL LETTER U WITH DIAERESIS
        // U+0169: "ũ" LATIN SMALL LETTER U WITH TILDE
        // U+016B: "ū" LATIN SMALL LETTER U WITH MACRON
        // U+016D: "ŭ" LATIN SMALL LETTER U WITH BREVE
        // U+016F: "ů" LATIN SMALL LETTER U WITH RING ABOVE
        // U+0171: "ű" LATIN SMALL LETTER U WITH DOUBLE ACUTE
        // U+0173: "ų" LATIN SMALL LETTER U WITH OGONEK
        /* morekeys_u */ "\u00F9,\u00FA,\u00FB,\u00FC,\u0169,\u016B,\u016D,\u016F,\u0171,\u0173",
        // U+00DF: "ß" LATIN SMALL LETTER SHARP S
        // U+015B: "ś" LATIN SMALL LETTER S WITH ACUTE
        // U+015D: "ŝ" LATIN SMALL LETTER S WITH CIRCUMFLEX
        // U+015F: "ş" LATIN SMALL LETTER S WITH CEDILLA
        // U+0161: "š" LATIN SMALL LETTER S WITH CARON
        // U+017F: "ſ" LATIN SMALL LETTER LONG S
        /* morekeys_s */ "\u00DF,\u015B,\u015D,\u015F,\u0161,\u017F",
        // U+00F1: "ñ" LATIN SMALL LETTER N WITH TILDE
        // U+0144: "ń" LATIN SMALL LETTER N WITH ACUTE
        // U+0146: "ņ" LATIN SMALL LETTER N WITH CEDILLA
        // U+0148: "ň" LATIN SMALL LETTER N WITH CARON
        // U+0149: "ŉ" LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
        // U+014B: "ŋ" LATIN SMALL LETTER ENG
        /* morekeys_n */ "\u00F1,\u0144,\u0146,\u0148,\u0149,\u014B",
        // U+00E7: "ç" LATIN SMALL LETTER C WITH CEDILLA
        // U+0107: "ć" LATIN SMALL LETTER C WITH ACUTE
        // U+0109: "ĉ" LATIN SMALL LETTER C WITH CIRCUMFLEX
        // U+010B: "ċ" LATIN SMALL LETTER C WITH DOT ABOVE
        // U+010D: "č" LATIN SMALL LETTER C WITH CARON
        /* morekeys_c */ "\u00E7,\u0107,\u0109,\u010B,\u010D",
        // U+00FD: "ý" LATIN SMALL LETTER Y WITH ACUTE
        // U+0177: "ŷ" LATIN SMALL LETTER Y WITH CIRCUMFLEX
        // U+00FF: "ÿ" LATIN SMALL LETTER Y WITH DIAERESIS
        // U+0133: "ĳ" LATIN SMALL LIGATURE IJ
        /* morekeys_y */ "\u00FD,\u0177,\u00FF,\u0133",
        // U+010F: "ď" LATIN SMALL LETTER D WITH CARON
        // U+0111: "đ" LATIN SMALL LETTER D WITH STROKE
        // U+00F0: "ð" LATIN SMALL LETTER ETH
        /* morekeys_d */ "\u010F,\u0111,\u00F0",
        // U+0155: "ŕ" LATIN SMALL LETTER R WITH ACUTE
        // U+0157: "ŗ" LATIN SMALL LETTER R WITH CEDILLA
        // U+0159: "ř" LATIN SMALL LETTER R WITH CARON
        /* morekeys_r */ "\u0155,\u0157,\u0159",
        // U+00FE: "þ" LATIN SMALL LETTER THORN
        // U+0163: "ţ" LATIN SMALL LETTER T WITH CEDILLA
        // U+0165: "ť" LATIN SMALL LETTER T WITH CARON
        // U+0167: "ŧ" LATIN SMALL LETTER T WITH STROKE
        /* morekeys_t */ "\u00FE,\u0163,\u0165,\u0167",
        // U+017A: "ź" LATIN SMALL LETTER Z WITH ACUTE
        // U+017C: "ż" LATIN SMALL LETTER Z WITH DOT ABOVE
        // U+017E: "ž" LATIN SMALL LETTER Z WITH CARON
        /* morekeys_z */ "\u017A,\u017C,\u017E",
        // U+0137: "ķ" LATIN SMALL LETTER K WITH CEDILLA
        // U+0138: "ĸ" LATIN SMALL LETTER KRA
        /* morekeys_k */ "\u0137,\u0138",
        // U+013A: "ĺ" LATIN SMALL LETTER L WITH ACUTE
        // U+013C: "ļ" LATIN SMALL LETTER L WITH CEDILLA
        // U+013E: "ľ" LATIN SMALL LETTER L WITH CARON
        // U+0140: "ŀ" LATIN SMALL LETTER L WITH MIDDLE DOT
        // U+0142: "ł" LATIN SMALL LETTER L WITH STROKE
        /* morekeys_l */ "\u013A,\u013C,\u013E,\u0140,\u0142",
        // U+011D: "ĝ" LATIN SMALL LETTER G WITH CIRCUMFLEX
        // U+011F: "ğ" LATIN SMALL LETTER G WITH BREVE
        // U+0121: "ġ" LATIN SMALL LETTER G WITH DOT ABOVE
        // U+0123: "ģ" LATIN SMALL LETTER G WITH CEDILLA
        /* morekeys_g */ "\u011D,\u011F,\u0121,\u0123",
        // U+0125: "ĥ" LATIN SMALL LETTER H WITH CIRCUMFLEX
        /* morekeys_h */ "\u0125",
        // U+0135: "ĵ" LATIN SMALL LETTER J WITH CIRCUMFLEX
        /* morekeys_j */ "\u0135",
        // U+0175: "ŵ" LATIN SMALL LETTER W WITH CIRCUMFLEX
        /* morekeys_w */ "\u0175",
    };

    private static final Object[] LOCALES_AND_TEXTS = {
    // "locale", TEXT_ARRAY,  /* numberOfNonNullText/lengthOf_TEXT_ARRAY localeName */
        "DEFAULT", TEXTS_DEFAULT, /* 176/176 DEFAULT */
        "ar"     , TEXTS_ar,    /*   2/ 25 Arabic */
        "as_IN"  , TEXTS_as_IN, /*  25/ 49 Assamese (India) */
        "bn_IN"  , TEXTS_bn_IN, /*  22/ 22 Bangla (India) */
        "en"     , TEXTS_en,    /*   8/ 36 English */
        "gu_IN"  , TEXTS_gu_IN, /*  27/ 28 Gujarati (India) */
        "hi"     , TEXTS_hi,    /*  27/ 28 Hindi */
        "hi_ZZ"  , TEXTS_hi_ZZ, /*   9/ 57 Hindi (Unknown Region) */
        "kn"     , TEXTS_kn,    /*  22/ 22 Kannada */
        "ml"     , TEXTS_ml,    /*  11/ 11 Malayalam */
        "mr"     , TEXTS_mr,    /*  23/ 23 Marathi */
        "my"     , TEXTS_my,    /*  22/ 22 Burmese */
        "ne"     , TEXTS_ne,    /*  27/ 28 Nepali */
        "or_IN"  , TEXTS_or_IN, /*  23/ 25 Odia (India) */
        "pa_IN"  , TEXTS_pa_IN, /*  22/ 22 Punjabi (India) */
        "ta_IN"  , TEXTS_ta_IN, /*  12/ 12 Tamil (India) */
        "ta_LK"  , TEXTS_ta_LK, /*   2/ 23 Tamil (Sri Lanka) */
        "ta_SG"  , TEXTS_ta_SG, /*   1/  1 Tamil (Singapore) */
        "te"     , TEXTS_te,    /*  22/ 22 Telugu */
        "zz"     , TEXTS_zz,    /*  19/ 47 Alphabet */
    };

    static {
        for (int index = 0; index < NAMES.length; index++) {
            sNameToIndexesMap.put(NAMES[index], index);
        }

        for (int i = 0; i < LOCALES_AND_TEXTS.length; i += 2) {
            final String locale = (String)LOCALES_AND_TEXTS[i];
            final String[] textsTable = (String[])LOCALES_AND_TEXTS[i + 1];
            sLocaleToTextsTableMap.put(locale, textsTable);
            sTextsTableToLocaleMap.put(textsTable, locale);
        }
    }
}
