"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.filePlugin = void 0;
const path_1 = __importDefault(require("path"));
const fs_1 = __importDefault(require("fs"));
const util_1 = require("util");
const mkdirp_1 = __importDefault(require("mkdirp"));
const readFile = (0, util_1.promisify)(fs_1.default.readFile);
const writeFile = (0, util_1.promisify)(fs_1.default.writeFile);
const unlink = (0, util_1.promisify)(fs_1.default.unlink);
const access = (0, util_1.promisify)(fs_1.default.access);
async function fileExists(filePath) {
    try {
        await access(filePath);
        return true;
    }
    catch (_a) {
        return false;
    }
}
function isObject(payload) {
    return payload != null && typeof payload === 'object';
}
function isWriteFilePayload(payload) {
    if (!isObject(payload))
        throw new Error('You must provide a payload object');
    if (typeof payload.path !== 'string')
        throw new Error('You must provide a path option');
    if (typeof payload.content !== 'string')
        throw new Error('You must provide a content option');
    return true;
}
function isReadFilePayload(payload) {
    if (!isObject(payload))
        throw new Error('You must provide a payload object');
    if (typeof payload.path !== 'string')
        throw new Error('You must provide a path option');
    return true;
}
function isRemoveFilePayload(payload) {
    if (!isObject(payload))
        throw new Error('You must provide a payload object');
    if (typeof payload.path !== 'string')
        throw new Error('You must provide a path option');
    return true;
}
function joinFilePath(testFile, relativePath) {
    if (path_1.default.isAbsolute(relativePath)) {
        throw new Error('file path must not be an absolute path.');
    }
    const dir = path_1.default.dirname(testFile);
    return path_1.default.join(dir, relativePath.split('/').join(path_1.default.sep));
}
function filePlugin() {
    return {
        name: 'file-commands',
        async executeCommand({ command, payload, session }) {
            if (command === 'write-file') {
                if (!isWriteFilePayload(payload)) {
                    throw new Error('You must provide a payload object');
                }
                const filePath = joinFilePath(session.testFile, payload.path);
                const fileDir = path_1.default.dirname(filePath);
                await (0, mkdirp_1.default)(fileDir);
                await writeFile(filePath, payload.content, payload.encoding || 'utf-8');
                return true;
            }
            if (command === 'read-file') {
                if (!isReadFilePayload(payload)) {
                    throw new Error('You must provide a payload object');
                }
                const filePath = joinFilePath(session.testFile, payload.path);
                if (await fileExists(filePath)) {
                    return readFile(filePath, payload.encoding || 'utf-8');
                }
                else {
                    return undefined;
                }
            }
            if (command === 'remove-file') {
                if (!isRemoveFilePayload(payload)) {
                    throw new Error('You must provide a payload object');
                }
                const filePath = joinFilePath(session.testFile, payload.path);
                await unlink(filePath);
                return true;
            }
        },
    };
}
exports.filePlugin = filePlugin;
//# sourceMappingURL=filePlugin.js.map