/**
 * Stores data serialized to a JSON file.
 */
export declare class Store<T> {
    /** Current state of the data. */
    protected _data?: T;
    /** Internal path for JSON file. */
    private readonly _path;
    /** Value used to initialize data for the first time. */
    private readonly _initial;
    /** A mutex to ensure that there aren't races while reading and writing files */
    private readonly _lock;
    /**
     * Creates a new store.
     *
     * @param path A unique filename to store this data.
     * @param id A unique filename to store this data.
     * @param initial An initial value to initialize data with.
     */
    constructor(path: string, id: string, initial: T);
    /**
     * Updates data by replacing it with the given value.
     * @param data New data to replace the previous one.
     */
    set(data: T): Promise<void>;
    /**
     * Returns the current data.
     *
     * When invoked for the first time, it will try to load previously stored data
     * from disk. If the file does not exist, the initial value provided to the
     * constructor is used.
     */
    get(): Promise<T>;
    /**
     * Updates data by passing it through the given function.
     * @param fn A function receiving the current data and returning new one.
     */
    update(fn: (current: T) => T): Promise<void>;
    /** Returns store to its initial state */
    clear(): Promise<void>;
    /** Gets the Date that the file was last modified */
    getModifiedDate(): Promise<Date | undefined>;
}
/**
 * Extends Store to throttle writes.
 */
export declare class BufferedWriteStore<T> extends Store<T> {
    private readonly _throttleTime;
    /** A write that hasn't been written to disk yet */
    private _pendingWrite;
    /**
     * Creates a new ThrottledStore.
     *
     * @param path A unique filename to store this data.
     * @param id A unique filename to store this data.
     * @param initial An initial value to initialize data with.
     * @param throttleTime The minimum time between writes
     */
    constructor(path: string, id: string, initial: T, _throttleTime?: number);
    /** @inheritdoc */
    set(data: T): Promise<void>;
    /** Writes the pending write to disk */
    private _writePending;
}
//# sourceMappingURL=store.d.ts.map