import { Middleware } from 'koa';
import { Options as NodeResolveOptions } from '@rollup/plugin-node-resolve';
import { PolyfillsLoaderConfig } from './utils/inject-polyfills-loader';
import { ResponseTransformer } from './middleware/response-transform';
import { TransformOptions } from '@babel/core';
import { Plugin } from './Plugin';
export interface CompressOptions {
    threshold?: number;
    filter?: (contentType: string) => boolean;
}
export interface Config {
    /** the port to run the server on */
    port?: number;
    /** the hostname to run the server on */
    hostname?: string;
    /**
     * if boolean: whether to open the browser on startup,
     * if string: what path to open the browser on. if it's a boolean, the open path is
     * determined by root dir + app index (if any)
     */
    open?: boolean | string;
    /**
     * file path to the index of your application, will serve this
     * index when requesting non-existing paths to enable SPA routing / history API fallback.
     * can be an absolute file path, or a path relative to the root dir
     */
    appIndex?: string;
    /**
     * root directory to set up the web server, any served files must
     * be within the scope of this folder
     */
    rootDir?: string;
    /**
     * Base path the app is served on. This path is only visible in
     * the browser, it is stripped from the request url before resolving files. Starts with a /
     * and ends with no/. For example: /my-app, /foo, /foo/bar
     */
    basePath?: string;
    /** Whether the server should compress responses. */
    compress?: boolean | CompressOptions;
    /** Middleware to inject into the server. */
    middlewares?: Middleware[];
    responseTransformers?: ResponseTransformer[];
    /** whether to log server startup */
    logStartup?: boolean;
    /** whether to log debug messages */
    debug?: boolean;
    /** Enable cors */
    cors?: boolean;
    /** whether to watch served files and reload the browser on change */
    watch?: boolean;
    /** whether to log errors to the browser */
    logErrorsToBrowser?: boolean;
    /** whether to run the server in http2, sets up https as well */
    http2?: boolean;
    /** path to local .key file to use for https */
    sslKey?: string;
    /** path to local .cert file to use for https */
    sslCert?: string;
    /**
     * compatibility mode for older browsers. Can be: "auto", "min",
     * "max" or "none". Defaults to "auto"
     */
    compatibility?: string;
    /** whether to resolve bare module imports using node resolve */
    nodeResolve?: boolean | NodeResolveOptions;
    /**
     * dedupe ensures only one
     * version of a module is ever resolved by resolving it from the root node_modules.
     */
    dedupeModules?: boolean | string[] | ((importee: string) => boolean);
    /** configuration for the polyfills loader */
    polyfillsLoader?: boolean | Partial<PolyfillsLoaderConfig>;
    /**
     * preserve symlinks when resolving modules. Default false,
     * which is the default node behavior.
     */
    preserveSymlinks?: boolean;
    /**
     * directories to resolve modules from when using nodeResolve
     * should be directory names, not paths to directories as node resolve will try to recursively
     * find directories with these names
     */
    moduleDirs?: string[];
    /** plugins to load */
    plugins?: Plugin[];
    /** Disables injecting event stream script */
    eventStream?: boolean;
    /** whether to use the user's .babelrc babel config */
    babel?: boolean;
    /** file extensions to run babel on */
    fileExtensions?: string[];
    /** files excluded from all babel compilation */
    babelExclude?: string[];
    /** files excluded from babel on modern browser */
    babelModernExclude?: string[];
    /** files excluded from module transformration */
    babelModuleExclude?: string[];
    /** files to include from user defined babel config */
    customBabelInclude?: string[];
    /** files to exclude from user defined babel config */
    customBabelExclude?: string[];
    /**
     * babel config to use, this is useful when you want to provide a
     * babel config from a tool, and don't want to require all users to use the same babel config
     */
    babelConfig?: TransformOptions;
    /**
     * callback called before the server is
     * started, the returned promise is awaited
     */
    onServerStart?: (config: ParsedConfig) => void | Promise<void>;
}
export interface ParsedConfig {
    port?: number;
    hostname?: string;
    openBrowser: boolean;
    openPath: string;
    appIndex?: string;
    appIndexDir?: string;
    basePath?: string;
    rootDir: string;
    logStartup: boolean;
    customMiddlewares: Middleware[];
    responseTransformers: ResponseTransformer[];
    onServerStart?: (config: ParsedConfig) => void | Promise<void>;
    cors: boolean;
    watch: boolean;
    logErrorsToBrowser: boolean;
    watchDebounce: number;
    http2: boolean;
    sslKey?: string;
    sslCert?: string;
    plugins: Plugin[];
    nodeResolve: boolean | NodeResolveOptions;
    polyfillsLoader: boolean;
    polyfillsLoaderConfig?: Partial<PolyfillsLoaderConfig>;
    readUserBabelConfig: boolean;
    compatibilityMode: string;
    compress: boolean | CompressOptions;
    customBabelConfig?: TransformOptions;
    fileExtensions: string[];
    babelExclude: string[];
    babelModernExclude: string[];
    babelModuleExclude: string[];
    customBabelInclude: string[];
    customBabelExclude: string[];
    eventStream: boolean;
}
/**
 * Creates dev server config with default settings
 */
export declare function createConfig(config: Partial<Config>): ParsedConfig;
