const tslib = require('tslib');
const utils = require('@sentry/utils');
const integrations = require('./integrations.js');
const core = require('@sentry/core');
const mode = require('./common/mode.js');

const Integrations = integrations.getIntegrations();
/** Fetches the SDK entry point for the current process */
function getEntryPoint() {
    try {
        return process.type === 'browser' ? utils.dynamicRequire(module, './main') : require('./renderer');
    }
    catch (e) {
        throw new Error(`Failed to automatically detect correct SDK entry point.

In the Electron main process you should import via:
import * as Sentry from '@sentry/electron/main';

In the Electron renderer process you should import via:
import * as Sentry from '@sentry/electron/renderer';`);
    }
}
/**
 * The Sentry Electron SDK Client.
 *
 * To use this SDK, call the {@link init} function as early as possible in the
 * entry modules. This applies to the main process as well as all renderer
 * processes or further sub processes you spawn. To set context information or
 * send manual events, use the provided methods.
 *
 * @example
 * const { init } = require('@sentry/electron');
 *
 * init({
 *   dsn: '__DSN__',
 *   // ...
 * });
 *
 * @example
 * import { configureScope } from '@sentry/electron';
 * configureScope((scope: Scope) => {
 *   scope.setExtra({ battery: 0.7 });
 *   scope.setTags({ user_mode: 'admin' });
 *   scope.setUser({ id: '4711' });
 * });
 *
 * @example
 * import { addBreadcrumb } from '@sentry/electron';
 * addBreadcrumb({
 *   message: 'My Breadcrumb',
 *   // ...
 * });
 *
 * @example
 * import * as Sentry from '@sentry/electron';
 * Sentry.captureMessage('Hello, world!');
 * Sentry.captureException(new Error('Good bye'));
 * Sentry.captureEvent({
 *   message: 'Manual',
 *   stacktrace: [
 *     // ...
 *   ],
 * });
 *
 * @see ElectronOptions for documentation on configuration options.
 */
function init(options) {
    // Filter out any EmptyIntegrations
    integrations.removeEmptyIntegrations(options);
    getEntryPoint().init(options);
}
/**
 * Call `close()` on the current client, if there is one. See {@link Client.close}.
 *
 * @param timeout Maximum time in ms the client should wait to flush its event queue before shutting down. Omitting this
 * parameter will cause the client to wait until all events are sent before disabling itself.
 * @returns A promise which resolves to `true` if the queue successfully drains before the timeout, or `false` if it
 * doesn't (or if there's no client defined).
 */
function close(timeout) {
    return tslib.__awaiter(this, void 0, void 0, function* () {
        const entryPoint = getEntryPoint();
        if (entryPoint.close) {
            return entryPoint.close(timeout);
        }
        throw new Error('The Electron SDK should be closed from the main process');
    });
}
/**
 * Call `flush()` on the current client, if there is one. See {@link Client.flush}.
 *
 * @param timeout Maximum time in ms the client should wait to flush its event queue. Omitting this parameter will cause
 * the client to wait until all events are sent before resolving the promise.
 * @returns A promise which resolves to `true` if the queue successfully drains before the timeout, or `false` if it
 * doesn't (or if there's no client defined).
 */
function flush(timeout) {
    return tslib.__awaiter(this, void 0, void 0, function* () {
        const entryPoint = getEntryPoint();
        if (entryPoint.flush) {
            return entryPoint.flush(timeout);
        }
        throw new Error('The Electron SDK should be flushed from the main process');
    });
}
/**
 * @deprecated Use `Anr` integration instead.
 *
 * ```js
 * import { init, Integrations } from '@sentry/electron';
 *
 * init({
 *   dsn: "__DSN__",
 *   integrations: [new Integrations.Anr({ captureStackTrace: true })],
 * });
 * ```
 */
// eslint-disable-next-line deprecation/deprecation
function enableMainProcessAnrDetection(options) {
    const entryPoint = getEntryPoint();
    if (entryPoint.enableMainProcessAnrDetection) {
        return entryPoint.enableMainProcessAnrDetection(options);
    }
    throw new Error('ANR detection should be started in the main process');
}

exports.Hub = core.Hub;
exports.Scope = core.Scope;
exports.addBreadcrumb = core.addBreadcrumb;
exports.addEventProcessor = core.addEventProcessor;
exports.addGlobalEventProcessor = core.addGlobalEventProcessor;
exports.addIntegration = core.addIntegration;
exports.captureCheckIn = core.captureCheckIn;
exports.captureEvent = core.captureEvent;
exports.captureException = core.captureException;
exports.captureMessage = core.captureMessage;
exports.configureScope = core.configureScope;
exports.continueTrace = core.continueTrace;
exports.createTransport = core.createTransport;
exports.extractTraceparentData = core.extractTraceparentData;
exports.getActiveSpan = core.getActiveSpan;
exports.getActiveTransaction = core.getActiveTransaction;
exports.getClient = core.getClient;
exports.getCurrentHub = core.getCurrentHub;
exports.getCurrentScope = core.getCurrentScope;
exports.getGlobalScope = core.getGlobalScope;
exports.getHubFromCarrier = core.getHubFromCarrier;
exports.getIsolationScope = core.getIsolationScope;
exports.lastEventId = core.lastEventId;
exports.makeMain = core.makeMain;
exports.metrics = core.metrics;
exports.runWithAsyncContext = core.runWithAsyncContext;
exports.setContext = core.setContext;
exports.setExtra = core.setExtra;
exports.setExtras = core.setExtras;
exports.setMeasurement = core.setMeasurement;
exports.setTag = core.setTag;
exports.setTags = core.setTags;
exports.setUser = core.setUser;
exports.spanStatusfromHttpCode = core.spanStatusfromHttpCode;
exports.startActiveSpan = core.startActiveSpan;
exports.startInactiveSpan = core.startInactiveSpan;
exports.startSpan = core.startSpan;
exports.startSpanManual = core.startSpanManual;
exports.startTransaction = core.startTransaction;
exports.trace = core.trace;
exports.withMonitor = core.withMonitor;
exports.withScope = core.withScope;
Object.defineProperty(exports, 'IPCMode', {
    enumerable: true,
    get: () => mode.IPCMode
});
exports.Integrations = Integrations;
exports.close = close;
exports.enableMainProcessAnrDetection = enableMainProcessAnrDetection;
exports.flush = flush;
exports.init = init;
//# sourceMappingURL=index.js.map
