"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _fs = _interopRequireDefault(require("fs"));

var _findUp = _interopRequireDefault(require("find-up"));

var _lodash = _interopRequireDefault(require("lodash.memoize"));

var _helpers = require("../helpers");

var _providers = require("../providers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Step 2) Logic that handles AST traversal
 * Does not handle looking up the API
 * Handles checking what kinds of eslint nodes should be linted
 *   Tells eslint to lint certain nodes  (lintCallExpression, lintMemberExpression, lintNewExpression)
 *   Gets protochain for the ESLint nodes the plugin is interested in
 */
function getName(node) {
  switch (node.type) {
    case "NewExpression":
      {
        return node.callee.name;
      }

    case "MemberExpression":
      {
        return node.object.name;
      }

    case "ExpressionStatement":
      {
        return node.expression.name;
      }

    case "CallExpression":
      {
        return node.callee.name;
      }

    default:
      throw new Error("not found");
  }
}

function generateErrorName(rule) {
  if (rule.name) return rule.name;
  if (rule.property) return `${rule.object}.${rule.property}()`;
  return rule.object;
}

const getPolyfillSet = (0, _lodash.default)(polyfillArrayJSON => new Set(JSON.parse(polyfillArrayJSON)));

function isPolyfilled(context, rule) {
  var _context$settings;

  if (!((_context$settings = context.settings) !== null && _context$settings !== void 0 && _context$settings.polyfills)) return false;
  const polyfills = getPolyfillSet(JSON.stringify(context.settings.polyfills));
  return (// v2 allowed users to select polyfills based off their caniuseId. This is
    polyfills.has(rule.id) || // no longer supported. Keeping this here to avoid breaking changes.
    polyfills.has(rule.protoChainId) || // Check if polyfill is provided (ex. `Promise.all`)
    polyfills.has(rule.protoChain[0]) // Check if entire API is polyfilled (ex. `Promise`)

  );
}

const items = [// Babel configs
"babel.config.json", "babel.config.js", "babel.config.cjs", ".babelrc", ".babelrc.json", ".babelrc.js", ".babelrc.cjs", // TS configs
"tsconfig.json"];
/**
 * Determine if a user has a TS or babel config. This is used to infer if a user is transpiling their code.
 * If transpiling code, do not lint ES APIs. We assume that all transpiled code is polyfilled.
 * @TODO Use @babel/core to find config. See https://github.com/babel/babel/discussions/11602
 * @param dir @
 */

function isUsingTranspiler(context) {
  var _context$parserOption;

  // If tsconfig config exists in parser options, assume transpilation
  if (((_context$parserOption = context.parserOptions) === null || _context$parserOption === void 0 ? void 0 : _context$parserOption.tsconfigRootDir) === true) return true;
  const dir = context.getFilename();

  const configPath = _findUp.default.sync(items, {
    cwd: dir
  });

  if (configPath) return true;

  const pkgPath = _findUp.default.sync("package.json", {
    cwd: dir
  }); // Check if babel property exists


  if (pkgPath) {
    const pkg = JSON.parse(_fs.default.readFileSync(pkgPath).toString());
    return !!pkg.babel;
  }

  return false;
}

/**
 * A small optimization that only lints APIs that are not supported by targeted browsers.
 * For example, if the user is targeting chrome 50, which supports the fetch API, it is
 * wasteful to lint calls to fetch.
 */
const getRulesForTargets = (0, _lodash.default)((targetsJSON, lintAllEsApis) => {
  const result = {
    CallExpression: [],
    NewExpression: [],
    MemberExpression: [],
    ExpressionStatement: []
  };
  const targets = JSON.parse(targetsJSON);

  _providers.nodes.filter(node => {
    return lintAllEsApis ? true : node.kind !== "es";
  }).forEach(node => {
    if (!node.getUnsupportedTargets(node, targets).length) return;
    result[node.astNodeType].push(node);
  });

  return result;
});
var _default = {
  meta: {
    docs: {
      description: "Ensure cross-browser API compatibility",
      category: "Compatibility",
      url: "https://github.com/amilajack/eslint-plugin-compat/blob/master/docs/rules/compat.md",
      recommended: true
    },
    type: "problem",
    schema: [{
      type: "string"
    }]
  },

  create(context) {
    var _context$settings2, _context$settings3, _context$settings4, _context$settings5, _context$settings5$po;

    // Determine lowest targets from browserslist config, which reads user's
    // package.json config section. Use config from eslintrc for testing purposes
    const browserslistConfig = ((_context$settings2 = context.settings) === null || _context$settings2 === void 0 ? void 0 : _context$settings2.browsers) || ((_context$settings3 = context.settings) === null || _context$settings3 === void 0 ? void 0 : _context$settings3.targets) || context.options[0];
    const lintAllEsApis = ((_context$settings4 = context.settings) === null || _context$settings4 === void 0 ? void 0 : _context$settings4.lintAllEsApis) === true || // Attempt to infer polyfilling of ES APIs from ts or babel config
    !((_context$settings5 = context.settings) !== null && _context$settings5 !== void 0 && (_context$settings5$po = _context$settings5.polyfills) !== null && _context$settings5$po !== void 0 && _context$settings5$po.includes("es:all")) && !isUsingTranspiler(context);
    const browserslistTargets = (0, _helpers.parseBrowsersListVersion)((0, _helpers.determineTargetsFromConfig)(context.getFilename(), browserslistConfig)); // Stringify to support memoization; browserslistConfig is always an array of new objects.

    const targetedRules = getRulesForTargets(JSON.stringify(browserslistTargets), lintAllEsApis);
    const errors = [];

    const handleFailingRule = (node, eslintNode) => {
      if (isPolyfilled(context, node)) return;
      errors.push({
        node: eslintNode,
        message: [generateErrorName(node), "is not supported in", node.getUnsupportedTargets(node, browserslistTargets).join(", ")].join(" ")
      });
    };

    const identifiers = new Set();
    return {
      CallExpression: _helpers.lintCallExpression.bind(null, context, handleFailingRule, targetedRules.CallExpression),
      NewExpression: _helpers.lintNewExpression.bind(null, context, handleFailingRule, targetedRules.NewExpression),
      ExpressionStatement: _helpers.lintExpressionStatement.bind(null, context, handleFailingRule, [...targetedRules.MemberExpression, ...targetedRules.CallExpression]),
      MemberExpression: _helpers.lintMemberExpression.bind(null, context, handleFailingRule, [...targetedRules.MemberExpression, ...targetedRules.CallExpression, ...targetedRules.NewExpression]),

      // Keep track of all the defined variables. Do not report errors for nodes that are not defined
      Identifier(node) {
        if (node.parent) {
          const {
            type
          } = node.parent;

          if (type === "Property" || // ex. const { Set } = require('immutable');
          type === "FunctionDeclaration" || // ex. function Set() {}
          type === "VariableDeclarator" || // ex. const Set = () => {}
          type === "ClassDeclaration" || // ex. class Set {}
          type === "ImportDefaultSpecifier" || // ex. import Set from 'set';
          type === "ImportSpecifier" || // ex. import {Set} from 'set';
          type === "ImportDeclaration" // ex. import {Set} from 'set';
          ) {
            identifiers.add(node.name);
          }
        }
      },

      "Program:exit": () => {
        // Get a map of all the variables defined in the root scope (not the global scope)
        // const variablesMap = context.getScope().childScopes.map(e => e.set)[0];
        errors.filter(error => !identifiers.has(getName(error.node))).forEach(node => context.report(node));
      }
    };
  }

};
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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