"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.getUnsupportedTargets = getUnsupportedTargets;
exports.isSupportedByMDN = isSupportedByMDN;

require("core-js/modules/es.array.flat.js");

var _astMetadataInferer = _interopRequireDefault(require("ast-metadata-inferer"));

var _semver = _interopRequireDefault(require("semver"));

var _helpers = require("../helpers");

var _constants = require("../constants");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const apis = _astMetadataInferer.default; // @TODO Import this type from ast-metadata-inferer after migrating this project to TypeScript

const mdnRecords = new Map(apis.map(e => [e.protoChainId, e]));

/**
 * Map ids of mdn targets to their "common/friendly" name
 */
const targetIdMappings = {
  chrome: "chrome",
  firefox: "firefox",
  opera: "opera",
  safari: "safari",
  safari_ios: "ios_saf",
  ie: "ie",
  edge_mobile: "ie_mob",
  edge: "edge",
  opera_android: "and_opera",
  chrome_android: "and_chrome",
  firefox_android: "and_firefox",
  webview_android: "and_webview",
  samsunginternet_android: "and_samsung",
  nodejs: "node"
};
const reversedTargetMappings = (0, _helpers.reverseTargetMappings)(targetIdMappings);
/**
 * Take a target's id and return it's full name by using `targetNameMappings`
 * ex. {target: and_ff, version: 40} => 'Android FireFox 40'
 */

function formatTargetNames(target) {
  return `${_constants.STANDARD_TARGET_NAME_MAPPING[target.target]} ${target.version}`;
}
/**
 * Convert '9' => '9.0.0'
 */


function customCoerce(version) {
  return version.length === 1 ? [version, 0, 0].join(".") : version;
}
/*
 * Return if MDN supports the API or not
 */


function isSupportedByMDN(node, {
  version,
  target: mdnTarget
}) {
  const target = reversedTargetMappings[mdnTarget]; // If no record could be found, return true. Rules might not
  // be found because they could belong to another provider

  if (!mdnRecords.has(node.protoChainId)) return true;
  const record = mdnRecords.get(node.protoChainId);
  if (!record || !record.compat.support) return true;
  const compatRecord = record.compat.support[target];
  if (!compatRecord) return true;
  if (!Array.isArray(compatRecord) && !("version_added" in compatRecord)) return true;
  const {
    version_added: versionAdded
  } = Array.isArray(compatRecord) ? compatRecord.find(e => "version_added" in e) : compatRecord; // If a version is true then it is supported but version is unsure

  if (typeof versionAdded === "boolean") return versionAdded;
  if (versionAdded === null) return true; // Special case for Safari TP: TP is always gte than any other releases

  if (target === "safari") {
    if (version === "TP") return true;
    if (versionAdded === "TP") return false;
  } // A browser supports an API if its version is greater than or equal
  // to the first version of the browser that API was added in


  const semverCurrent = _semver.default.coerce(customCoerce(version));

  const semverAdded = _semver.default.coerce(customCoerce(versionAdded)); // semver.coerce() might be null for non-semvers (other than Safari TP)
  // Just warn and treat features as supported here for now to avoid lint from
  // crashing


  if (!semverCurrent) {
    // eslint-disable-next-line no-console
    console.warn(`eslint-plugin-compat: A non-semver target "${target} ${version}" matched for the feature ${node.protoChainId}, skipping. You're welcome to submit this log to https://github.com/amilajack/eslint-plugin-compat/issues for analysis.`);
    return true;
  }

  if (!versionAdded) {
    // eslint-disable-next-line no-console
    console.warn(`eslint-plugin-compat: The feature ${node.protoChainId} is supported since a non-semver target "${target} ${versionAdded}", skipping. You're welcome to submit this log to https://github.com/amilajack/eslint-plugin-compat/issues for analysis.`);
    return true;
  }

  return _semver.default.gte(semverCurrent, semverAdded);
}
/**
 * Return an array of all unsupported targets
 */


function getUnsupportedTargets(node, targets) {
  return targets.filter(target => !isSupportedByMDN(node, target)).map(formatTargetNames);
}

function getMetadataName(metadata) {
  switch (metadata.protoChain.length) {
    case 1:
      {
        return metadata.protoChain[0];
      }

    default:
      return `${metadata.protoChain.join(".")}()`;
  }
}

const MdnProvider = apis // Create entries for each ast node type
.map(metadata => metadata.astNodeTypes.map(astNodeType => ({ ...metadata,
  name: getMetadataName(metadata),
  id: metadata.protoChainId,
  protoChainId: metadata.protoChainId,
  astNodeType,
  object: metadata.protoChain[0],
  // @TODO Handle cases where 'prototype' is in protoChain
  property: metadata.protoChain[1]
}))) // Flatten the array of arrays
.flat() // Add rule and target support logic for each entry
.map(rule => ({ ...rule,
  getUnsupportedTargets
}));
var _default = MdnProvider;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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