"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.getUnsupportedTargets = getUnsupportedTargets;

var lite = _interopRequireWildcard(require("caniuse-lite"));

var _constants = require("../constants");

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/**
 * Take a target's id and return it's full name by using `targetNameMappings`
 * ex. {target: and_ff, version: 40} => 'Android FireFox 40'
 */
function formatTargetNames(target) {
  const name = _constants.STANDARD_TARGET_NAME_MAPPING[target.target] || target.target;
  return `${name} ${target.version}`;
}
/**
 * Check if a browser version is in the range format
 * ex. 10.0-10.2
 */


function versionIsRange(version) {
  return version.includes("-");
}
/**
 * Parse version from caniuse and compare with parsed version from browserslist.
 */


function areVersionsEqual(targetVersion, statsVersion) {
  return targetVersion === parseFloat(statsVersion);
}
/*
 * Check the CanIUse database to see if targets are supported
 *
 * If no record could be found, return true. Rules might not
 * be found because they could belong to another provider
 */


function isSupportedByCanIUse(node, {
  version,
  target,
  parsedVersion
}) {
  const data = lite.feature(lite.features[node.caniuseId]);
  if (!data) return true;
  const {
    stats
  } = data;
  if (!(target in stats)) return true;
  const targetStats = stats[target];

  if (typeof version === "string" && versionIsRange(version)) {
    return Object.keys(targetStats).some(statsVersion => versionIsRange(statsVersion) && areVersionsEqual(parsedVersion, statsVersion) ? !targetStats[statsVersion].includes("y") : true);
  } // @TODO: This assumes that all versions are included in the cainuse db. If this is incorrect,
  //        this will return false negatives. To properly do this, we have to to range comparisons.
  //        Ex. given query for 50 and only version 40 exists in db records, return true


  if (!(version in targetStats)) return true;
  if (!targetStats[version]) return true;
  return targetStats[version].includes("y");
}
/**
 * Return an array of all unsupported targets
 */


function getUnsupportedTargets(node, targets) {
  return targets.filter(target => !isSupportedByCanIUse(node, target)).map(formatTargetNames);
}

const CanIUseProvider = [// new ServiceWorker()
{
  caniuseId: "serviceworkers",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "ServiceWorker"
}, {
  caniuseId: "serviceworkers",
  astNodeType: _constants.AstNodeTypes.MemberExpression,
  object: "navigator",
  property: "serviceWorker"
}, // document.querySelector()
{
  caniuseId: "queryselector",
  astNodeType: _constants.AstNodeTypes.MemberExpression,
  object: "document",
  property: "querySelector"
}, // IntersectionObserver
{
  caniuseId: "intersectionobserver",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "IntersectionObserver"
}, // ResizeObserver
{
  caniuseId: "resizeobserver",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "ResizeObserver"
}, // PaymentRequest
{
  caniuseId: "payment-request",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "PaymentRequest"
}, // Promises
{
  caniuseId: "promises",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "Promise"
}, {
  caniuseId: "promises",
  astNodeType: _constants.AstNodeTypes.MemberExpression,
  object: "Promise",
  property: "resolve"
}, {
  caniuseId: "promises",
  astNodeType: _constants.AstNodeTypes.MemberExpression,
  object: "Promise",
  property: "all"
}, {
  caniuseId: "promises",
  astNodeType: _constants.AstNodeTypes.MemberExpression,
  object: "Promise",
  property: "race"
}, {
  caniuseId: "promises",
  astNodeType: _constants.AstNodeTypes.MemberExpression,
  object: "Promise",
  property: "reject"
}, // fetch
{
  caniuseId: "fetch",
  astNodeType: _constants.AstNodeTypes.CallExpression,
  object: "fetch"
}, // document.currentScript()
{
  caniuseId: "document-currentscript",
  astNodeType: _constants.AstNodeTypes.MemberExpression,
  object: "document",
  property: "currentScript"
}, // URL
{
  caniuseId: "url",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "URL"
}, // URLSearchParams
{
  caniuseId: "urlsearchparams",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "URLSearchParams"
}, // performance.now()
{
  caniuseId: "high-resolution-time",
  astNodeType: _constants.AstNodeTypes.MemberExpression,
  object: "performance",
  property: "now"
}, {
  caniuseId: "typedarrays",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "TypedArray"
}, {
  caniuseId: "typedarrays",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "Int8Array"
}, {
  caniuseId: "typedarrays",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "Uint8Array"
}, {
  caniuseId: "typedarrays",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "Uint8ClampedArray"
}, {
  caniuseId: "typedarrays",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "Int16Array"
}, {
  caniuseId: "typedarrays",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "Uint16Array"
}, {
  caniuseId: "typedarrays",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "Int32Array"
}, {
  caniuseId: "typedarrays",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "Uint32Array"
}, {
  caniuseId: "typedarrays",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "Float32Array"
}, {
  caniuseId: "typedarrays",
  astNodeType: _constants.AstNodeTypes.NewExpression,
  object: "Float64Array"
}].map(rule => ({ ...rule,
  getUnsupportedTargets,
  id: rule.property ? `${rule.object}.${rule.property}` : rule.object,
  protoChainId: rule.property ? `${rule.object}.${rule.property}` : rule.object,
  protoChain: rule.property ? [rule.object, rule.property] : [rule.object]
}));
var _default = CanIUseProvider;
exports.default = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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