"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.serveTestRunnerHtmlPlugin = void 0;
const dev_server_core_1 = require("@web/dev-server-core");
const constants_1 = require("../../utils/constants");
const utils_1 = require("../utils");
const trackBrowserLogs_1 = require("./trackBrowserLogs");
const iframeModePage = `
<!DOCTYPE html>
<html>
  <head>
    <style>
      iframe {
        position: absolute;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        border: none;
      }
    </style>
  </head>
  <body></body>
</html>
`;
async function getManualListItem(config, context, testFile) {
    const testImportPath = await (0, utils_1.createTestFileImportPath)(config, context, testFile);
    const displayedPath = testImportPath.split('?')[0].substring(1);
    const pagename = displayedPath.endsWith('.html') ? displayedPath : '/';
    const href = `${pagename}?${constants_1.PARAM_TEST_FILE}=${encodeURIComponent(testImportPath)}`;
    return `<li><a href="${href}">${displayedPath}</a></li>`;
}
async function createManualDebugPage(config, context, testFiles) {
    const listItems = (await Promise.all(testFiles.map(file => getManualListItem(config, context, file)))).join('\n');
    return `<!DOCTYPE html>
<html>
  <head>
    <style>
      p {
        max-width: 600px;
      }
    </style>
  </head>

  <body>
    <h1>Web Test Runner</h1>
    <p>Select a file to debug manually in a custom browser not controlled by Web Test Runner.</p>
    
    <p>
      Advanced functionalities such commands for changing viewport and screenshots don't work here. 
      Use the regular debug option to debug in a controlled browser.
    </p>

    <h2>Test files</h2>
    <ul>
      ${listItems}
    </ul>
  </body>
</html>
`;
}
function getInjectIndex(html) {
    const headIndex = html.indexOf('<head>');
    if (headIndex !== -1)
        return '<head>'.length + headIndex;
    const bodyIndex = html.indexOf('<body>');
    if (bodyIndex !== -1)
        return '<body>'.length + bodyIndex;
    return 0;
}
function injectRuntime(html, config, browserLogs, preloadedModule) {
    const preloadLink = preloadedModule
        ? `<link rel="preload" as="script" crossorigin="anonymous" href="${preloadedModule}">`
        : '';
    const configScript = `<script type="module">window.__WTR_CONFIG__ = ${JSON.stringify(config)}</script>`;
    const injectedHtml = `${preloadLink}${browserLogs ? `${trackBrowserLogs_1.trackBrowserLogs}` : ''}${configScript}`;
    const injectLocation = getInjectIndex(html);
    return `${html.substring(0, injectLocation)}${injectedHtml}${html.substring(injectLocation)}`;
}
function createTestRunnerHtml(testFrameworkImport, config, groupConfig) {
    let body;
    if (groupConfig === null || groupConfig === void 0 ? void 0 : groupConfig.testRunnerHtml) {
        // there is a group in scope, regular test or debug
        body = groupConfig.testRunnerHtml(testFrameworkImport, config, groupConfig);
    }
    else if (config.testRunnerHtml) {
        // there is no group in scope, ex. when manually debugging
        body = config.testRunnerHtml(testFrameworkImport, config);
    }
    else {
        // no user defined test runner HTML
        body = `<!DOCTYPE html><html><head></head><body><script type="module" src="${testFrameworkImport}"></script></body></html>`;
    }
    return { body, type: 'html' };
}
function serveTestRunnerHtmlPlugin(config, testFiles, sessions, testFrameworkImport) {
    return {
        name: 'wtr-test-runner-html',
        async serve(context) {
            var _a;
            if (!testFrameworkImport) {
                throw new Error('Cannot test javascript files without a testFramework configured.');
            }
            if (context.path === '/') {
                const { searchParams } = context.URL;
                if (searchParams.has(constants_1.PARAM_TEST_FILE)) {
                    return createTestRunnerHtml(testFrameworkImport, config);
                }
                const sessionId = searchParams.get(constants_1.PARAM_SESSION_ID);
                if (sessionId) {
                    const session = (_a = sessions.get(sessionId)) !== null && _a !== void 0 ? _a : sessions.getDebug(sessionId);
                    if (!session) {
                        throw new Error(`Could not find session ${sessionId}`);
                    }
                    return createTestRunnerHtml(testFrameworkImport, config, session.group);
                }
                if (searchParams.get('mode') === 'iframe') {
                    return {
                        type: 'html',
                        body: iframeModePage,
                    };
                }
                return {
                    type: 'html',
                    body: await createManualDebugPage(config, context, testFiles),
                };
            }
        },
        async transform(context) {
            var _a, _b, _c;
            if (!context.response.is('html')) {
                return;
            }
            const isTestRunnerHtml = context.path === '/';
            if (!isTestRunnerHtml &&
                !testFiles.includes((0, dev_server_core_1.getRequestFilePath)(context.url, config.rootDir))) {
                return;
            }
            const { searchParams } = context.URL;
            const sessionId = searchParams.get(constants_1.PARAM_SESSION_ID);
            if (sessionId) {
                const session = (_a = sessions.get(sessionId)) !== null && _a !== void 0 ? _a : sessions.getDebug(sessionId);
                if (!session) {
                    throw new Error(`Could not find session ${sessionId}`);
                }
                const testFile = await (0, utils_1.createTestFileImportPath)(config, context, session.testFile, sessionId);
                const runtimeConfig = {
                    testFile,
                    watch: !!config.watch,
                    debug: session.debug,
                    testFrameworkConfig: (_b = config.testFramework) === null || _b === void 0 ? void 0 : _b.config,
                };
                context.body = injectRuntime(context.body, runtimeConfig, !!config.browserLogs, isTestRunnerHtml ? testFile : undefined);
                return;
            }
            const testFile = searchParams.get(constants_1.PARAM_TEST_FILE);
            if (testFile) {
                const runtimeConfig = {
                    testFile,
                    watch: !!config.watch,
                    debug: true,
                    testFrameworkConfig: (_c = config.testFramework) === null || _c === void 0 ? void 0 : _c.config,
                };
                context.body = injectRuntime(context.body, runtimeConfig, !!config.browserLogs, isTestRunnerHtml ? testFile : undefined);
                return;
            }
        },
    };
}
exports.serveTestRunnerHtmlPlugin = serveTestRunnerHtmlPlugin;
//# sourceMappingURL=serveTestRunnerHtmlPlugin.js.map