"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.testRunnerApiPlugin = void 0;
const dev_server_core_1 = require("@web/dev-server-core");
const co_body_1 = __importDefault(require("co-body"));
const constants_1 = require("../../../utils/constants");
const TestSessionStatus_1 = require("../../../test-session/TestSessionStatus");
const parseBrowserResult_1 = require("./parseBrowserResult");
const createSourceMapFunction_1 = require("./createSourceMapFunction");
function createMapBrowserUrl(rootDir) {
    return function mapBrowserUrl(url) {
        if (url.pathname.startsWith('/__web-test-runner__/test-framework')) {
            return url.pathname.substring('/__web-test-runner__/test-framework'.length);
        }
        return (0, dev_server_core_1.getRequestFilePath)(url.href, rootDir);
    };
}
class TestRunnerApiPlugin {
    constructor(config, testRunner, sessions, plugins) {
        this.name = 'test-runner-api';
        this.injectWebSocket = true;
        /** key: session id, value: browser url */
        this.testSessionUrls = new Map();
        this.config = config;
        this.testRunner = testRunner;
        this.sessions = sessions;
        this.plugins = plugins;
        this.mapBrowserUrl = createMapBrowserUrl(config.rootDir);
        this.sourceMapFunction = (0, createSourceMapFunction_1.createSourceMapFunction)(this.config.protocol, this.config.hostname, this.config.port);
        this.testRunner.on('test-run-started', ({ testRun }) => {
            if (testRun !== 0) {
                // create a new source map function to clear the cached source maps
                this.sourceMapFunction = (0, createSourceMapFunction_1.createSourceMapFunction)(this.config.protocol, this.config.hostname, this.config.port);
            }
        });
    }
    getSession(sessionId) {
        const session = this.sessions.get(sessionId) || this.sessions.getDebug(sessionId);
        if (!session) {
            throw new Error(`Session id ${sessionId} not found`);
        }
        return session;
    }
    parseSessionMessage(data) {
        if (typeof data.sessionId === 'string') {
            return { message: data, session: this.getSession(data.sessionId) };
        }
        throw new Error('Missing sessionId in browser websocket message.');
    }
    async transform(context) {
        if (context.response.is('html')) {
            const sessionId = context.URL.searchParams.get(constants_1.PARAM_SESSION_ID);
            if (!sessionId) {
                return;
            }
            try {
                this.testSessionUrls.set(sessionId, `${this.config.protocol}//${this.config.hostname}:${this.config.port}${context.url}`);
            }
            catch (error) {
                this.config.logger.error('Error while creating test file import path');
                this.config.logger.error(error);
                context.status = 500;
            }
        }
    }
    serverStart({ webSockets }) {
        webSockets.on('message', async ({ webSocket, data }) => {
            if (!data.type.startsWith('wtr-')) {
                return;
            }
            const { session, message } = this.parseSessionMessage(data);
            if (data.type === 'wtr-session-started') {
                if (!data.testFile && !session.debug) {
                    // handle started only when session isn't debug or manual
                    this._onSessionStarted(session);
                    webSocket.on('close', async () => {
                        this._waitForDisconnect(session.id, session.testRun);
                    });
                }
                return;
            }
            if (data.type === 'wtr-session-finished') {
                this._onSessionFinished(data, session, message);
                return;
            }
            if (data.type === 'wtr-command') {
                this._onCommand(webSocket, session, message);
                return;
            }
        });
    }
    async serve(context) {
        if (context.path === '/__web-test-runner__/wtr-legacy-browser-api') {
            const data = await co_body_1.default.json(context);
            const { session, message } = this.parseSessionMessage(data);
            if (data.type === 'wtr-session-started') {
                if (!data.testFile && !session.debug) {
                    // handle started only when session isn't debug or manual
                    this._onSessionStarted(session);
                    setTimeout(() => {
                        this._waitForDisconnect(session.id, session.testRun);
                    }, this.config.testsFinishTimeout);
                }
                return { body: 'OK', type: 'text' };
            }
            if (data.type === 'wtr-session-finished') {
                this._onSessionFinished(data, session, message);
                return { body: 'OK', type: 'text' };
            }
            return { body: 'Commands are not supported', type: 'text', status: 500 };
        }
    }
    _onSessionStarted(session) {
        if (session.status !== TestSessionStatus_1.SESSION_STATUS.INITIALIZING) {
            this._onMultiInitialized(session);
            return;
        }
        // mark the session as started
        this.sessions.updateStatus(session, TestSessionStatus_1.SESSION_STATUS.TEST_STARTED);
    }
    async _onSessionFinished(rawData, session, message) {
        if (session.debug)
            return;
        if (typeof message.result !== 'object') {
            throw new Error('Missing result in session-finished message.');
        }
        if (!rawData.userAgent || typeof rawData.userAgent !== 'string') {
            throw new Error('Missing userAgent in session-finished message.');
        }
        const result = await (0, parseBrowserResult_1.parseBrowserResult)(this.config, this.mapBrowserUrl, this.sourceMapFunction, rawData.userAgent, message.result);
        this.sessions.updateStatus(Object.assign(Object.assign({}, session), result), TestSessionStatus_1.SESSION_STATUS.TEST_FINISHED);
    }
    async _onCommand(webSocket, session, message) {
        var _a;
        const { id, command, payload } = message;
        if (typeof id !== 'number')
            throw new Error('Missing message id.');
        if (typeof command !== 'string')
            throw new Error('A command name must be provided.');
        for (const plugin of this.plugins) {
            try {
                const result = await ((_a = plugin.executeCommand) === null || _a === void 0 ? void 0 : _a.call(plugin, { command, payload, session }));
                if (result != null) {
                    webSocket.send(JSON.stringify({
                        type: 'message-response',
                        id,
                        response: { executed: true, result },
                    }));
                    return;
                }
            }
            catch (error) {
                this.config.logger.error(error);
                webSocket.send(JSON.stringify({ type: 'message-response', id, error: error.message }));
                return;
            }
        }
        // no command was matched
        webSocket.send(JSON.stringify({ type: 'message-response', id, response: { executed: false } }));
    }
    /**
     * Waits for web socket to become disconnected, and checks after disconnect if it was expected
     * or if some error occurred.
     */
    async _waitForDisconnect(sessionId, testRun) {
        const session = this.sessions.get(sessionId);
        if ((session === null || session === void 0 ? void 0 : session.testRun) !== testRun) {
            // a new testrun was started
            return;
        }
        if ((session === null || session === void 0 ? void 0 : session.status) !== TestSessionStatus_1.SESSION_STATUS.TEST_STARTED) {
            // websocket closed after finishing the tests, this is expected
            return;
        }
        // the websocket disconnected while the tests were still running, this can happen for many reasons.
        // we wait 2000ms (magic number) to let other handlers come up with a more specific error message
        await new Promise(r => setTimeout(r, 2000));
        const updatedSession = this.sessions.get(sessionId);
        if ((updatedSession === null || updatedSession === void 0 ? void 0 : updatedSession.status) !== TestSessionStatus_1.SESSION_STATUS.TEST_STARTED) {
            // something else handled the disconnect
            return;
        }
        if ((updatedSession === null || updatedSession === void 0 ? void 0 : updatedSession.testRun) !== testRun) {
            // a new testrun was started
            return;
        }
        const startUrl = this.testSessionUrls.get(updatedSession.id);
        const currentUrl = await updatedSession.browser.getBrowserUrl(updatedSession.id);
        if (!currentUrl || startUrl !== currentUrl) {
            this._setSessionFailed(updatedSession, `Tests were interrupted because the page navigated to ${currentUrl ? currentUrl : 'another origin'}. ` +
                'This can happen when clicking a link, submitting a form or interacting with window.location.');
        }
        else {
            this._setSessionFailed(updatedSession, 'Tests were interrupted because the browser disconnected.');
        }
    }
    _onMultiInitialized(session) {
        this._setSessionFailed(session, 'Tests were interrupted because the page was reloaded. ' +
            'This can happen when clicking a link, submitting a form or interacting with window.location.');
    }
    _setSessionFailed(session, message) {
        var _a;
        this.sessions.updateStatus(Object.assign(Object.assign({}, session), { errors: [...((_a = session.errors) !== null && _a !== void 0 ? _a : []), { message }] }), TestSessionStatus_1.SESSION_STATUS.TEST_FINISHED);
    }
}
function testRunnerApiPlugin(config, testRunner, sessions, plugins) {
    return new TestRunnerApiPlugin(config, testRunner, sessions, plugins);
}
exports.testRunnerApiPlugin = testRunnerApiPlugin;
//# sourceMappingURL=testRunnerApiPlugin.js.map