"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TestScheduler = void 0;
const createSessionUrl_1 = require("./createSessionUrl");
const TestSessionStatus_1 = require("../test-session/TestSessionStatus");
const async_1 = require("../utils/async");
const TestSessionTimeoutHandler_1 = require("./TestSessionTimeoutHandler");
class TestScheduler {
    constructor(config, sessions, browsers) {
        this.finishedBrowsers = new Set();
        this.stopPromises = new Set();
        this.config = config;
        this.sessions = sessions;
        this.browsers = [...browsers].sort((a, b) => (a.__experimentalWindowFocus__ ? 1 : 0) - (b.__experimentalWindowFocus__ ? 1 : 0));
        this.timeoutHandler = new TestSessionTimeoutHandler_1.TestSessionTimeoutHandler(this.config, this.sessions, this);
        this.browserStartTimeoutMsg =
            `The browser was unable to create and start a test page after ${this.config.browserStartTimeout}ms. ` +
                'You can increase this timeout with the browserStartTimeout option.';
        sessions.on('session-status-updated', async (session) => {
            if (session.status === TestSessionStatus_1.SESSION_STATUS.TEST_STARTED) {
                this.timeoutHandler.waitForTestsFinished(session.testRun, session.id);
                return;
            }
            if (session.status === TestSessionStatus_1.SESSION_STATUS.TEST_FINISHED) {
                // the session finished executing tests, close the browser page
                await this.stopSession(session);
                return;
            }
            if (session.status === TestSessionStatus_1.SESSION_STATUS.FINISHED) {
                this.timeoutHandler.clearTimeoutsForSession(session);
                setTimeout(() => {
                    // run next scheduled after a timeout, so that other actors on status change can
                    // do their work first
                    this.runNextScheduled();
                });
            }
        });
    }
    /**
     * Schedules a session for execution. Execution is batched, the session
     * will be queued until there is a browser page available.
     */
    schedule(testRun, sessionsToSchedule) {
        for (const session of sessionsToSchedule) {
            this.sessions.updateStatus(Object.assign(Object.assign({}, session), { testRun, request404s: [] }), TestSessionStatus_1.SESSION_STATUS.SCHEDULED);
        }
        this.runNextScheduled();
    }
    stop() {
        this.timeoutHandler.clearAllTimeouts();
        return Promise.all(Array.from(this.stopPromises));
    }
    /** Runs the next batch of scheduled sessions, if any. */
    runNextScheduled() {
        var _a;
        let runningBrowsers = 0;
        for (const browser of this.browsers) {
            if (runningBrowsers >= 1 && browser.__experimentalWindowFocus__) {
                return;
            }
            if (runningBrowsers >= this.config.concurrentBrowsers) {
                // do not boot up more than the allowed concurrent browsers
                return;
            }
            const unfinishedCount = this.getUnfinishedSessions(browser).length;
            if (unfinishedCount > 0) {
                // this browser has not finished running all tests
                runningBrowsers += 1;
                const runningCount = this.getRunningSessions(browser).length;
                let maxBudget;
                if (browser.__experimentalWindowFocus__) {
                    maxBudget = 1;
                }
                else {
                    maxBudget = (_a = browser.concurrency) !== null && _a !== void 0 ? _a : this.config.concurrency;
                }
                const runBudget = Math.max(0, maxBudget - runningCount);
                if (runBudget !== 0) {
                    // we have budget to schedule new sessions for this browser
                    const allScheduled = this.getScheduledSessions(browser);
                    const toRun = allScheduled.slice(0, runBudget);
                    for (const session of toRun) {
                        this.startSession(session);
                    }
                }
            }
        }
    }
    async startSession(session) {
        const updatedSession = Object.assign(Object.assign({}, session), { status: TestSessionStatus_1.SESSION_STATUS.INITIALIZING });
        this.sessions.updateStatus(updatedSession, TestSessionStatus_1.SESSION_STATUS.INITIALIZING);
        let browserStarted = false;
        // browser should be started within the specified milliseconds
        const timeoutId = setTimeout(() => {
            if (!browserStarted && !this.timeoutHandler.isStale(updatedSession)) {
                this.setSessionFailed(this.sessions.get(updatedSession.id), {
                    message: this.browserStartTimeoutMsg,
                });
            }
        }, this.config.browserStartTimeout);
        this.timeoutHandler.addTimeoutId(updatedSession.id, timeoutId);
        try {
            await (0, async_1.withTimeout)(updatedSession.browser.startSession(updatedSession.id, (0, createSessionUrl_1.createSessionUrl)(this.config, updatedSession)), this.browserStartTimeoutMsg, this.config.browserStartTimeout);
            // when the browser started, wait for session to ping back on time
            this.timeoutHandler.waitForTestsStarted(updatedSession.testRun, updatedSession.id);
        }
        catch (e) {
            const error = e;
            if (this.timeoutHandler.isStale(updatedSession)) {
                // something else has changed the test session, such as a the browser timeout
                // or a re-run in watch mode. in that was we just log the error
                if (error.message !== this.browserStartTimeoutMsg) {
                    this.config.logger.error(error);
                }
            }
            else {
                this.setSessionFailed(updatedSession, { message: error.message, stack: error.stack });
            }
        }
        finally {
            browserStarted = true;
        }
    }
    setSessionFailed(session, ...errors) {
        this.stopSession(session, errors);
    }
    async stopSession(session, errors = []) {
        var _a, _b;
        if (this.timeoutHandler.isStale(session)) {
            return;
        }
        const sessionErrors = [...errors];
        const updatedSession = Object.assign({}, session);
        try {
            if (session.browser.isActive(session.id)) {
                const { testCoverage, errors } = await (0, async_1.withTimeout)(session.browser.stopSession(session.id), 'Timed out stopping the browser page', this.config.testsFinishTimeout);
                updatedSession.errors = [...((_a = updatedSession.errors) !== null && _a !== void 0 ? _a : []), ...(errors !== null && errors !== void 0 ? errors : [])];
                updatedSession.testCoverage = testCoverage;
            }
        }
        catch (error) {
            sessionErrors.push(error);
        }
        finally {
            if (sessionErrors.length > 0) {
                // merge with existing erors
                updatedSession.errors = [...((_b = updatedSession.errors) !== null && _b !== void 0 ? _b : []), ...sessionErrors];
                updatedSession.passed = false;
            }
            this.sessions.updateStatus(updatedSession, TestSessionStatus_1.SESSION_STATUS.FINISHED);
            const remaining = this.getUnfinishedSessions(session.browser);
            if (!this.config.watch &&
                !this.finishedBrowsers.has(session.browser) &&
                remaining.length === 0) {
                if (session.browser.stop) {
                    this.finishedBrowsers.add(session.browser);
                    const stopPromise = session.browser
                        .stop()
                        .catch(error => {
                        console.error(error);
                    })
                        .then(() => {
                        this.stopPromises.delete(stopPromise);
                    });
                    this.stopPromises.add(stopPromise);
                }
            }
        }
    }
    getScheduledSessions(browser) {
        return Array.from(this.sessions.filtered(s => s.browser === browser && s.status === TestSessionStatus_1.SESSION_STATUS.SCHEDULED));
    }
    getRunningSessions(browser) {
        return Array.from(this.sessions.filtered(s => s.browser === browser &&
            [
                TestSessionStatus_1.SESSION_STATUS.INITIALIZING,
                TestSessionStatus_1.SESSION_STATUS.TEST_STARTED,
                TestSessionStatus_1.SESSION_STATUS.TEST_FINISHED,
            ].includes(s.status)));
    }
    getUnfinishedSessions(browser) {
        return Array.from(this.sessions.filtered(s => s.browser === browser && s.status !== TestSessionStatus_1.SESSION_STATUS.FINISHED));
    }
}
exports.TestScheduler = TestScheduler;
//# sourceMappingURL=TestScheduler.js.map