"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getInputData = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const glob_1 = __importDefault(require("glob"));
const utils_1 = require("../utils");
const normalizeInputOptions_1 = require("./normalizeInputOptions");
const extractModulesAndAssets_1 = require("./extract/extractModulesAndAssets");
function resolveGlob(fromGlob, opts) {
    const files = glob_1.default.sync(fromGlob, Object.assign(Object.assign({}, opts), { absolute: true }));
    return (files
        // filter out directories
        .filter(filePath => !fs_1.default.lstatSync(filePath).isDirectory()));
}
function getName(filePath, rootDir, flattenOutput = true) {
    if (flattenOutput) {
        return path_1.default.basename(filePath);
    }
    return path_1.default.relative(rootDir, filePath);
}
function createInputData(params) {
    const { name, html, rootDir, filePath, extractAssets, absolutePathPrefix } = params;
    const htmlFilePath = filePath ? filePath : path_1.default.resolve(rootDir, name);
    const result = extractModulesAndAssets_1.extractModulesAndAssets({
        html,
        htmlFilePath,
        rootDir,
        extractAssets,
        absolutePathPrefix,
    });
    return {
        html: result.htmlWithoutModules,
        name,
        inlineModules: result.inlineModules,
        moduleImports: [...result.moduleImports, ...result.inlineModules],
        assets: result.assets,
        filePath,
    };
}
function getInputData(pluginOptions, rollupInput) {
    var _a, _b;
    const { rootDir = process.cwd(), flattenOutput, extractAssets = true, absolutePathPrefix, exclude: ignore, } = pluginOptions;
    const allInputs = normalizeInputOptions_1.normalizeInputOptions(pluginOptions, rollupInput);
    const result = [];
    for (const input of allInputs) {
        if (typeof input.html === 'string') {
            const name = (_a = input.name) !== null && _a !== void 0 ? _a : 'index.html';
            const data = createInputData({
                name,
                html: input.html,
                rootDir,
                extractAssets,
                absolutePathPrefix,
            });
            result.push(data);
        }
        else if (typeof input.path === 'string') {
            const filePaths = resolveGlob(input.path, { cwd: rootDir, ignore });
            if (filePaths.length === 0) {
                throw new Error(`Could not find any HTML files for pattern: ${input.path}, resolved relative to ${rootDir}`);
            }
            for (const filePath of filePaths) {
                const name = (_b = input.name) !== null && _b !== void 0 ? _b : getName(filePath, rootDir, flattenOutput);
                const html = fs_1.default.readFileSync(filePath, 'utf-8');
                const data = createInputData({
                    name,
                    html,
                    rootDir,
                    filePath,
                    extractAssets,
                    absolutePathPrefix,
                });
                result.push(data);
            }
        }
        else {
            throw utils_1.createError('An input must specify either a path or html.');
        }
    }
    for (const input of result) {
        if (result.filter(r => r.name === input.name).length !== 1) {
            throw utils_1.createError(`Found multiple HTML inputs configured with the same name, ` +
                'or with no name which defaults to index.html. Set a unique name on the' +
                'input option.');
        }
    }
    return result;
}
exports.getInputData = getInputData;
//# sourceMappingURL=getInputData.js.map