"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.injectPolyfillsLoader = void 0;
const parse5_1 = require("parse5");
const parse5_utils_1 = require("@web/parse5-utils");
const createPolyfillsLoader_1 = require("./createPolyfillsLoader");
const utils_1 = require("./utils");
function injectImportMapPolyfill(headAst, originalScript, type) {
    const systemJsScript = parse5_utils_1.createScript({ type }, parse5_utils_1.getTextContent(originalScript));
    parse5_utils_1.insertBefore(headAst, systemJsScript, originalScript);
}
function findImportMapScripts(document) {
    const scripts = parse5_utils_1.findElements(document, script => parse5_utils_1.getAttribute(script, 'type') === 'importmap');
    const inline = [];
    const external = [];
    for (const script of scripts) {
        if (parse5_utils_1.getAttribute(script, 'src')) {
            external.push(script);
        }
        else {
            inline.push(script);
        }
    }
    return { inline, external };
}
function injectImportMapPolyfills(documentAst, headAst, cfg) {
    const importMapScripts = findImportMapScripts(documentAst);
    if (importMapScripts.external.length === 0 && importMapScripts.inline.length === 0) {
        return;
    }
    const polyfillSystemJs = utils_1.hasFileOfType(cfg, utils_1.fileTypes.SYSTEMJS);
    const importMaps = [...importMapScripts.external, ...importMapScripts.inline];
    importMaps.forEach(originalScript => {
        if (polyfillSystemJs) {
            injectImportMapPolyfill(headAst, originalScript, 'systemjs-importmap');
        }
    });
}
function injectLoaderScript(bodyAst, polyfillsLoader, cfg) {
    let loaderScript;
    if (cfg.externalLoaderScript) {
        const loaderScriptFile = polyfillsLoader.polyfillFiles.find(f => f.path.endsWith('loader.js'));
        if (!loaderScriptFile) {
            throw new Error('Missing polyfills loader script file');
        }
        loaderScript = parse5_utils_1.createScript({ src: loaderScriptFile.path });
    }
    else {
        loaderScript = parse5_utils_1.createScript({}, polyfillsLoader.code);
    }
    parse5_utils_1.appendChild(bodyAst, loaderScript);
}
function injectPrefetchLinks(headAst, cfg) {
    for (const file of cfg.modern.files) {
        const { path } = file;
        const href = path.startsWith('.') || path.startsWith('/') ? path : `./${path}`;
        if (file.type === utils_1.fileTypes.MODULE) {
            parse5_utils_1.appendChild(headAst, parse5_utils_1.createElement('link', {
                rel: 'preload',
                href,
                as: 'script',
                crossorigin: 'anonymous',
            }));
        }
        else {
            parse5_utils_1.appendChild(headAst, parse5_utils_1.createElement('link', { rel: 'preload', href, as: 'script' }));
        }
    }
}
/**
 * Transforms an index.html file, injecting a polyfills loader for
 * compatibility with older browsers.
 */
async function injectPolyfillsLoader(htmlString, cfg) {
    const documentAst = parse5_1.parse(htmlString);
    const headAst = parse5_utils_1.findElement(documentAst, e => parse5_utils_1.getTagName(e) === 'head');
    const bodyAst = parse5_utils_1.findElement(documentAst, e => parse5_utils_1.getTagName(e) === 'body');
    if (!headAst || !bodyAst) {
        throw new Error(`Invalid index.html: missing <head> or <body>`);
    }
    const polyfillsLoader = await createPolyfillsLoader_1.createPolyfillsLoader(cfg);
    if (polyfillsLoader === null) {
        return { htmlString, polyfillFiles: [] };
    }
    if (cfg.preload) {
        injectPrefetchLinks(headAst, cfg);
    }
    injectImportMapPolyfills(documentAst, headAst, cfg);
    injectLoaderScript(bodyAst, polyfillsLoader, cfg);
    return {
        htmlString: parse5_1.serialize(documentAst),
        polyfillFiles: polyfillsLoader.polyfillFiles,
    };
}
exports.injectPolyfillsLoader = injectPolyfillsLoader;
//# sourceMappingURL=injectPolyfillsLoader.js.map