"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.storybookPlugin = void 0;
const dev_server_core_1 = require("@web/dev-server-core");
const storybook_addon_markdown_docs_1 = require("storybook-addon-markdown-docs");
const createManagerHtml_1 = require("../shared/html/createManagerHtml");
const createPreviewHtml_1 = require("../shared/html/createPreviewHtml");
const readStorybookConfig_1 = require("../shared/config/readStorybookConfig");
const validatePluginConfig_1 = require("../shared/config/validatePluginConfig");
const findStories_1 = require("../shared/stories/findStories");
const transformMdxToCsf_1 = require("../shared/mdx/transformMdxToCsf");
const injectExportsOrder_1 = require("../shared/stories/injectExportsOrder");
const regexpReplaceWebsocket = /<!-- injected by web-dev-server -->(.|\s)*<\/script>/m;
function storybookPlugin(pluginConfig) {
    validatePluginConfig_1.validatePluginConfig(pluginConfig);
    const storybookConfig = readStorybookConfig_1.readStorybookConfig(pluginConfig);
    let serverConfig;
    let storyImports = [];
    let storyFilePaths = [];
    return {
        name: 'storybook',
        serverStart(args) {
            serverConfig = args.config;
        },
        resolveMimeType(context) {
            if (context.URL.searchParams.get('story') !== 'true') {
                return;
            }
            if (context.path.endsWith('.mdx') || context.path.endsWith('.md')) {
                return 'js';
            }
        },
        async transform(context) {
            if (typeof context.body !== 'string') {
                return;
            }
            if (context.path === '/') {
                // replace the injected websocket script to avoid reloading the manager in watch mode
                context.body = context.body.replace(regexpReplaceWebsocket, '');
                return;
            }
            if (context.URL.searchParams.get('story') !== 'true') {
                return;
            }
            const filePath = dev_server_core_1.getRequestFilePath(context.url, serverConfig.rootDir);
            if (context.path.endsWith('.mdx')) {
                context.body = await transformMdxToCsf_1.transformMdxToCsf(context.body, filePath);
            }
            if (context.path.endsWith('.md')) {
                context.body = await storybook_addon_markdown_docs_1.mdjsToCsf(context.body, filePath, pluginConfig.type);
            }
            if (storyFilePaths.includes(filePath)) {
                // inject story order, note that MDX and MD and fall through to this as well
                context.body = await injectExportsOrder_1.injectExportsOrder(context.body, filePath);
            }
        },
        async serve(context) {
            if (context.path === '/') {
                return { type: 'html', body: createManagerHtml_1.createManagerHtml(storybookConfig, serverConfig.rootDir) };
            }
            if (context.path === '/iframe.html') {
                ({ storyImports, storyFilePaths } = await findStories_1.findStories(serverConfig.rootDir, storybookConfig.mainJsPath, storybookConfig.mainJs.stories));
                storyImports = storyImports.map(i => `${i}?story=true`);
                return {
                    type: 'html',
                    body: createPreviewHtml_1.createPreviewHtml(pluginConfig, storybookConfig, serverConfig.rootDir, storyImports),
                };
            }
        },
    };
}
exports.storybookPlugin = storybookPlugin;
//# sourceMappingURL=storybookPlugin.js.map