"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createMiddleware = void 0;
const koa_etag_1 = __importDefault(require("koa-etag"));
const basePathMiddleware_1 = require("../middleware/basePathMiddleware");
const etagCacheMiddleware_1 = require("../middleware/etagCacheMiddleware");
const historyApiFallbackMiddleware_1 = require("../middleware/historyApiFallbackMiddleware");
const pluginMimeTypeMiddleware_1 = require("../middleware/pluginMimeTypeMiddleware");
const pluginServeMiddleware_1 = require("../middleware/pluginServeMiddleware");
const pluginTransformMiddleware_1 = require("../middleware/pluginTransformMiddleware");
const watchServedFilesMiddleware_1 = require("../middleware/watchServedFilesMiddleware");
const pluginFileParsedMiddleware_1 = require("../middleware/pluginFileParsedMiddleware");
const serveFilesMiddleware_1 = require("../middleware/serveFilesMiddleware");
/**
 * Creates middlewares based on the given configuration. The middlewares can be
 * used by a koa server using `app.use()`:
 */
function createMiddleware(config, logger, fileWatcher) {
    var _a, _b;
    const middlewares = [];
    middlewares.push(async (ctx, next) => {
        logger.debug(`Receiving request: ${ctx.url}`);
        await next();
        logger.debug(`Responding to request: ${ctx.url} with status ${ctx.status}`);
    });
    // strips a base path from requests
    if (config.basePath) {
        middlewares.push(basePathMiddleware_1.basePathMiddleware(config.basePath));
    }
    // adds custom user's middlewares
    for (const m of (_a = config.middleware) !== null && _a !== void 0 ? _a : []) {
        middlewares.push(m);
    }
    // watch files that are served
    middlewares.push(watchServedFilesMiddleware_1.watchServedFilesMiddleware(fileWatcher, config.rootDir));
    // serves 304 responses if resource hasn't changed
    middlewares.push(etagCacheMiddleware_1.etagCacheMiddleware());
    // adds etag headers for caching
    middlewares.push(koa_etag_1.default());
    // serves index.html for non-file requests for SPA routing
    if (config.appIndex) {
        middlewares.push(historyApiFallbackMiddleware_1.historyApiFallbackMiddleware(config.appIndex, config.rootDir, logger));
    }
    const plugins = (_b = config.plugins) !== null && _b !== void 0 ? _b : [];
    middlewares.push(pluginFileParsedMiddleware_1.pluginFileParsedMiddleware(plugins));
    middlewares.push(pluginTransformMiddleware_1.pluginTransformMiddleware(logger, config, fileWatcher));
    middlewares.push(pluginMimeTypeMiddleware_1.pluginMimeTypeMiddleware(logger, plugins));
    middlewares.push(pluginServeMiddleware_1.pluginServeMiddleware(logger, plugins));
    middlewares.push(...serveFilesMiddleware_1.serveFilesMiddleware(config.rootDir));
    return middlewares;
}
exports.createMiddleware = createMiddleware;
//# sourceMappingURL=createMiddleware.js.map