import { createPerformanceSpans } from '../util/createPerformanceSpans.js';
import { shouldFilterRequest } from '../util/shouldFilterRequest.js';

// From sentry-javascript
// e.g. https://github.com/getsentry/sentry-javascript/blob/c7fc025bf9fa8c073fdb56351808ce53909fbe45/packages/utils/src/instrument.ts#L180

function handleXhr(handlerData) {
  if (handlerData.xhr.__sentry_own_request__) {
    // Taken from sentry-javascript
    // Only capture non-sentry requests
    return null;
  }

  if (handlerData.startTimestamp) {
    // TODO: See if this is still needed
    handlerData.xhr.__sentry_xhr__ = handlerData.xhr.__sentry_xhr__ || {};
    handlerData.xhr.__sentry_xhr__.startTimestamp = handlerData.startTimestamp;
  }

  if (!handlerData.endTimestamp) {
    return null;
  }

  const { method, url, status_code: statusCode } = handlerData.xhr.__sentry_xhr__ || {};

  if (url === undefined) {
    return null;
  }

  const timestamp = handlerData.xhr.__sentry_xhr__
    ? handlerData.xhr.__sentry_xhr__.startTimestamp || 0
    : handlerData.endTimestamp;

  return {
    type: 'resource.xhr',
    name: url,
    start: timestamp / 1000,
    end: handlerData.endTimestamp / 1000,
    data: {
      method,
      statusCode,
    },
  };
}

/**
 * Returns a listener to be added to `addInstrumentationHandler('xhr', listener)`.
 */
function handleXhrSpanListener(replay) {
  return (handlerData) => {
    if (!replay.isEnabled()) {
      return;
    }

    const result = handleXhr(handlerData);

    if (result === null) {
      return;
    }

    if (shouldFilterRequest(replay, result.name)) {
      return;
    }

    replay.addUpdate(() => {
      createPerformanceSpans(replay, [result]);
      // Returning true will cause `addUpdate` to not flush
      // We do not want network requests to cause a flush. This will prevent
      // recurring/polling requests from keeping the replay session alive.
      return true;
    });
  };
}

export { handleXhrSpanListener };
//# sourceMappingURL=handleXhr.js.map
