Object.defineProperty(exports, '__esModule', { value: true });

const core = require('@sentry/core');
const constants = require('../constants.js');
const sendReplayRequest = require('./sendReplayRequest.js');

/**
 * Finalize and send the current replay event to Sentry
 */
async function sendReplay(
  replayData,
  retryConfig = {
    count: 0,
    interval: constants.RETRY_BASE_INTERVAL,
  },
) {
  const { recordingData, options } = replayData;

  // short circuit if there's no events to upload (this shouldn't happen as _runFlush makes this check)
  if (!recordingData.length) {
    return;
  }

  try {
    await sendReplayRequest.sendReplayRequest(replayData);
    return true;
  } catch (err) {
    if (err instanceof sendReplayRequest.RateLimitError || err instanceof sendReplayRequest.TransportStatusCodeError) {
      throw err;
    }

    // Capture error for every failed replay
    core.setContext('Replays', {
      _retryCount: retryConfig.count,
    });

    if ((typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && options._experiments && options._experiments.captureExceptions) {
      core.captureException(err);
    }

    // If an error happened here, it's likely that uploading the attachment
    // failed, we'll can retry with the same events payload
    if (retryConfig.count >= constants.RETRY_MAX_COUNT) {
      throw new Error(`${constants.UNABLE_TO_SEND_REPLAY} - max retries exceeded`);
    }

    // will retry in intervals of 5, 10, 30
    retryConfig.interval *= ++retryConfig.count;

    return await new Promise((resolve, reject) => {
      setTimeout(async () => {
        try {
          await sendReplay(replayData, retryConfig);
          resolve(true);
        } catch (err) {
          reject(err);
        }
      }, retryConfig.interval);
    });
  }
}

exports.sendReplay = sendReplay;
//# sourceMappingURL=sendReplay.js.map
