Object.defineProperty(exports, '__esModule', { value: true });

const core = require('@sentry/core');
const constants = require('./constants.js');
const replay = require('./replay.js');
const isBrowser = require('./util/isBrowser.js');

const MEDIA_SELECTORS = 'img,image,svg,path,rect,area,video,object,picture,embed,map,audio';

let _initialized = false;

/**
 * The main replay integration class, to be passed to `init({  integrations: [] })`.
 */
class Replay  {
  /**
   * @inheritDoc
   */
   static __initStatic() {this.id = 'Replay';}

  /**
   * @inheritDoc
   */
   __init() {this.name = Replay.id;}

  /**
   * Options to pass to `rrweb.record()`
   */

   constructor({
    flushMinDelay = constants.DEFAULT_FLUSH_MIN_DELAY,
    flushMaxDelay = constants.DEFAULT_FLUSH_MAX_DELAY,
    initialFlushDelay = constants.INITIAL_FLUSH_DELAY,
    stickySession = true,
    useCompression = true,
    sessionSampleRate,
    errorSampleRate,
    maskAllText,
    maskTextSelector,
    maskAllInputs = true,
    blockAllMedia = true,
    _experiments = {},
    blockClass = 'sentry-block',
    ignoreClass = 'sentry-ignore',
    maskTextClass = 'sentry-mask',
    blockSelector = '[data-sentry-block]',
    ..._recordingOptions
  } = {}) {;Replay.prototype.__init.call(this);
    this._recordingOptions = {
      maskAllInputs,
      blockClass,
      ignoreClass,
      maskTextClass,
      maskTextSelector,
      blockSelector,
      ..._recordingOptions,
    };

    this._options = {
      flushMinDelay,
      flushMaxDelay,
      stickySession,
      initialFlushDelay,
      sessionSampleRate: constants.DEFAULT_SESSION_SAMPLE_RATE,
      errorSampleRate: constants.DEFAULT_ERROR_SAMPLE_RATE,
      useCompression,
      maskAllText: typeof maskAllText === 'boolean' ? maskAllText : !maskTextSelector,
      blockAllMedia,
      _experiments,
    };

    if (typeof sessionSampleRate === 'number') {
      // eslint-disable-next-line
      console.warn(
        `[Replay] You are passing \`sessionSampleRate\` to the Replay integration.
This option is deprecated and will be removed soon.
Instead, configure \`replaysSessionSampleRate\` directly in the SDK init options, e.g.:
Sentry.init({ replaysSessionSampleRate: ${sessionSampleRate} })`,
      );

      this._options.sessionSampleRate = sessionSampleRate;
    }

    if (typeof errorSampleRate === 'number') {
      // eslint-disable-next-line
      console.warn(
        `[Replay] You are passing \`errorSampleRate\` to the Replay integration.
This option is deprecated and will be removed soon.
Instead, configure \`replaysOnErrorSampleRate\` directly in the SDK init options, e.g.:
Sentry.init({ replaysOnErrorSampleRate: ${errorSampleRate} })`,
      );

      this._options.errorSampleRate = errorSampleRate;
    }

    if (this._options.maskAllText) {
      // `maskAllText` is a more user friendly option to configure
      // `maskTextSelector`. This means that all nodes will have their text
      // content masked.
      this._recordingOptions.maskTextSelector = constants.MASK_ALL_TEXT_SELECTOR;
    }

    if (this._options.blockAllMedia) {
      // `blockAllMedia` is a more user friendly option to configure blocking
      // embedded media elements
      this._recordingOptions.blockSelector = !this._recordingOptions.blockSelector
        ? MEDIA_SELECTORS
        : `${this._recordingOptions.blockSelector},${MEDIA_SELECTORS}`;
    }

    if (this._isInitialized && isBrowser.isBrowser()) {
      throw new Error('Multiple Sentry Session Replay instances are not supported');
    }

    this._isInitialized = true;
  }

  /** If replay has already been initialized */
   get _isInitialized() {
    return _initialized;
  }

  /** Update _isInitialized */
   set _isInitialized(value) {
    _initialized = value;
  }

  /**
   * We previously used to create a transaction in `setupOnce` and it would
   * potentially create a transaction before some native SDK integrations have run
   * and applied their own global event processor. An example is:
   * https://github.com/getsentry/sentry-javascript/blob/b47ceafbdac7f8b99093ce6023726ad4687edc48/packages/browser/src/integrations/useragent.ts
   *
   * So we call `replay.setup` in next event loop as a workaround to wait for other
   * global event processors to finish. This is no longer needed, but keeping it
   * here to avoid any future issues.
   */
   setupOnce() {
    if (!isBrowser.isBrowser()) {
      return;
    }

    this._setup();

    // XXX: See method comments above
    setTimeout(() => this.start());
  }

  /**
   * Initializes the plugin.
   *
   * Creates or loads a session, attaches listeners to varying events (DOM,
   * PerformanceObserver, Recording, Sentry SDK, etc)
   */
   start() {
    if (!this._replay) {
      return;
    }

    this._replay.start();
  }

  /**
   * Currently, this needs to be manually called (e.g. for tests). Sentry SDK
   * does not support a teardown
   */
   stop() {
    if (!this._replay) {
      return;
    }

    this._replay.stop();
  }

  /** Setup the integration. */
   _setup() {
    // Client is not available in constructor, so we need to wait until setupOnce
    this._loadReplayOptionsFromClient();

    this._replay = new replay.ReplayContainer({
      options: this._options,
      recordingOptions: this._recordingOptions,
    });
  }

  /** Parse Replay-related options from SDK options */
   _loadReplayOptionsFromClient() {
    const client = core.getCurrentHub().getClient();
    const opt = client && (client.getOptions() );

    if (opt && typeof opt.replaysSessionSampleRate === 'number') {
      this._options.sessionSampleRate = opt.replaysSessionSampleRate;
    }

    if (opt && typeof opt.replaysOnErrorSampleRate === 'number') {
      this._options.errorSampleRate = opt.replaysOnErrorSampleRate;
    }
  }
} Replay.__initStatic();

exports.Replay = Replay;
//# sourceMappingURL=integration.js.map
