var {
  _optionalChain
} = require('@sentry/utils/cjs/buildPolyfills');

Object.defineProperty(exports, '__esModule', { value: true });

const inspector = require('inspector');
const lru_map = require('lru_map');

/**
 * Promise API is available as `Experimental` and in Node 19 only.
 *
 * Callback-based API is `Stable` since v14 and `Experimental` since v8.
 * Because of that, we are creating our own `AsyncSession` class.
 *
 * https://nodejs.org/docs/latest-v19.x/api/inspector.html#promises-api
 * https://nodejs.org/docs/latest-v14.x/api/inspector.html
 */
class AsyncSession extends inspector.Session  {
  /** @inheritdoc */
   configureAndConnect(onPause) {
    this.connect();
    this.on('Debugger.paused', onPause);
    this.post('Debugger.enable');
    // We only want to pause on uncaught exceptions
    this.post('Debugger.setPauseOnExceptions', { state: 'uncaught' });
  }

  /** @inheritdoc */
   async getLocalVariables(objectId) {
    const props = await this._getProperties(objectId);
    const unrolled = {};

    for (const prop of props) {
      if (_optionalChain([prop, 'optionalAccess', _ => _.value, 'optionalAccess', _2 => _2.objectId]) && _optionalChain([prop, 'optionalAccess', _3 => _3.value, 'access', _4 => _4.className]) === 'Array') {
        unrolled[prop.name] = await this._unrollArray(prop.value.objectId);
      } else if (_optionalChain([prop, 'optionalAccess', _5 => _5.value, 'optionalAccess', _6 => _6.objectId]) && _optionalChain([prop, 'optionalAccess', _7 => _7.value, 'optionalAccess', _8 => _8.className]) === 'Object') {
        unrolled[prop.name] = await this._unrollObject(prop.value.objectId);
      } else if (_optionalChain([prop, 'optionalAccess', _9 => _9.value, 'optionalAccess', _10 => _10.value]) || _optionalChain([prop, 'optionalAccess', _11 => _11.value, 'optionalAccess', _12 => _12.description])) {
        unrolled[prop.name] = prop.value.value || `<${prop.value.description}>`;
      }
    }

    return unrolled;
  }

  /**
   * Gets all the PropertyDescriptors of an object
   */
   _getProperties(objectId) {
    return new Promise((resolve, reject) => {
      this.post(
        'Runtime.getProperties',
        {
          objectId,
          ownProperties: true,
        },
        (err, params) => {
          if (err) {
            reject(err);
          } else {
            resolve(params.result);
          }
        },
      );
    });
  }

  /**
   * Unrolls an array property
   */
   async _unrollArray(objectId) {
    const props = await this._getProperties(objectId);
    return props
      .filter(v => v.name !== 'length' && !isNaN(parseInt(v.name, 10)))
      .sort((a, b) => parseInt(a.name, 10) - parseInt(b.name, 10))
      .map(v => _optionalChain([v, 'optionalAccess', _13 => _13.value, 'optionalAccess', _14 => _14.value]));
  }

  /**
   * Unrolls an object property
   */
   async _unrollObject(objectId) {
    const props = await this._getProperties(objectId);
    return props
      .map(v => [v.name, _optionalChain([v, 'optionalAccess', _15 => _15.value, 'optionalAccess', _16 => _16.value])])
      .reduce((obj, [key, val]) => {
        obj[key] = val;
        return obj;
      }, {} );
  }
}

// Add types for the exception event data

/** Could this be an anonymous function? */
function isAnonymous(name) {
  return name !== undefined && ['', '?', '<anonymous>'].includes(name);
}

/** Do the function names appear to match? */
function functionNamesMatch(a, b) {
  return a === b || (isAnonymous(a) && isAnonymous(b));
}

/** Creates a unique hash from stack frames */
function hashFrames(frames) {
  if (frames === undefined) {
    return;
  }

  // Only hash the 10 most recent frames (ie. the last 10)
  return frames.slice(-10).reduce((acc, frame) => `${acc},${frame.function},${frame.lineno},${frame.colno}`, '');
}

/**
 * We use the stack parser to create a unique hash from the exception stack trace
 * This is used to lookup vars when the exception passes through the event processor
 */
function hashFromStack(stackParser, stack) {
  if (stack === undefined) {
    return undefined;
  }

  return hashFrames(stackParser(stack, 1));
}

/**
 * Adds local variables to exception frames
 */
class LocalVariables  {
   static __initStatic() {this.id = 'LocalVariables';}

    __init() {this.name = LocalVariables.id;}

    __init2() {this._cachedFrames = new lru_map.LRUMap(20);}

   constructor(_options = {},   _session = new AsyncSession()) {;this._session = _session;LocalVariables.prototype.__init.call(this);LocalVariables.prototype.__init2.call(this);}

  /**
   * @inheritDoc
   */
   setupOnce(addGlobalEventProcessor, getCurrentHub) {
    this._setup(addGlobalEventProcessor, _optionalChain([getCurrentHub, 'call', _17 => _17(), 'access', _18 => _18.getClient, 'call', _19 => _19(), 'optionalAccess', _20 => _20.getOptions, 'call', _21 => _21()]));
  }

  /** Setup in a way that's easier to call from tests */
   _setup(
    addGlobalEventProcessor,
    clientOptions,
  ) {
    if (_optionalChain([clientOptions, 'optionalAccess', _22 => _22._experiments, 'optionalAccess', _23 => _23.includeStackLocals])) {
      this._session.configureAndConnect(ev =>
        this._handlePaused(clientOptions.stackParser, ev ),
      );

      addGlobalEventProcessor(async event => this._addLocalVariables(event));
    }
  }

  /**
   * Handle the pause event
   */
   async _handlePaused(
    stackParser,
    { params: { reason, data, callFrames } },
  ) {
    if (reason !== 'exception' && reason !== 'promiseRejection') {
      return;
    }

    // data.description contains the original error.stack
    const exceptionHash = hashFromStack(stackParser, _optionalChain([data, 'optionalAccess', _24 => _24.description]));

    if (exceptionHash == undefined) {
      return;
    }

    const framePromises = callFrames.map(async ({ scopeChain, functionName, this: obj }) => {
      const localScope = scopeChain.find(scope => scope.type === 'local');

      const fn = obj.className === 'global' ? functionName : `${obj.className}.${functionName}`;

      if (_optionalChain([localScope, 'optionalAccess', _25 => _25.object, 'access', _26 => _26.objectId]) === undefined) {
        return { function: fn };
      }

      const vars = await this._session.getLocalVariables(localScope.object.objectId);

      return { function: fn, vars };
    });

    // We add the un-awaited promise to the cache rather than await here otherwise the event processor
    // can be called before we're finished getting all the vars
    this._cachedFrames.set(exceptionHash, Promise.all(framePromises));
  }

  /**
   * Adds local variables event stack frames.
   */
   async _addLocalVariables(event) {
    for (const exception of _optionalChain([event, 'optionalAccess', _27 => _27.exception, 'optionalAccess', _28 => _28.values]) || []) {
      await this._addLocalVariablesToException(exception);
    }

    return event;
  }

  /**
   * Adds local variables to the exception stack frames.
   */
   async _addLocalVariablesToException(exception) {
    const hash = hashFrames(_optionalChain([exception, 'optionalAccess', _29 => _29.stacktrace, 'optionalAccess', _30 => _30.frames]));

    if (hash === undefined) {
      return;
    }

    // Check if we have local variables for an exception that matches the hash
    // delete is identical to get but also removes the entry from the cache
    const cachedFrames = await this._cachedFrames.delete(hash);

    if (cachedFrames === undefined) {
      return;
    }

    const frameCount = _optionalChain([exception, 'access', _31 => _31.stacktrace, 'optionalAccess', _32 => _32.frames, 'optionalAccess', _33 => _33.length]) || 0;

    for (let i = 0; i < frameCount; i++) {
      // Sentry frames are in reverse order
      const frameIndex = frameCount - i - 1;

      // Drop out if we run out of frames to match up
      if (!_optionalChain([exception, 'optionalAccess', _34 => _34.stacktrace, 'optionalAccess', _35 => _35.frames, 'optionalAccess', _36 => _36[frameIndex]]) || !cachedFrames[i]) {
        break;
      }

      if (
        // We need to have vars to add
        cachedFrames[i].vars === undefined ||
        // We're not interested in frames that are not in_app because the vars are not relevant
        exception.stacktrace.frames[frameIndex].in_app === false ||
        // The function names need to match
        !functionNamesMatch(exception.stacktrace.frames[frameIndex].function, cachedFrames[i].function)
      ) {
        continue;
      }

      exception.stacktrace.frames[frameIndex].vars = cachedFrames[i].vars;
    }
  }
}LocalVariables.__initStatic();

exports.LocalVariables = LocalVariables;
//# sourceMappingURL=localvariables.js.map
