import type { Breadcrumb, ReplayRecordingMode, TransportMakeRequestResponse } from '@sentry/types';
import type { AddUpdateCallback, AllPerformanceEntry, EventBuffer, InstrumentationTypeBreadcrumb, InternalEventContext, PopEventContext, RecordingEvent, RecordingOptions, ReplayContainer as ReplayContainerInterface, ReplayPluginOptions, SendReplay, Session } from './types';
/**
 * The main replay container class, which holds all the state and methods for recording and sending replays.
 */
export declare class ReplayContainer implements ReplayContainerInterface {
    eventBuffer: EventBuffer | null;
    /**
     * List of PerformanceEntry from PerformanceObserver
     */
    performanceEvents: AllPerformanceEntry[];
    session: Session | undefined;
    /**
     * Recording can happen in one of two modes:
     * * session: Record the whole session, sending it continuously
     * * error: Always keep the last 60s of recording, and when an error occurs, send it immediately
     */
    recordingMode: ReplayRecordingMode;
    /**
     * Options to pass to `rrweb.record()`
     */
    private readonly _recordingOptions;
    private readonly _options;
    private _performanceObserver;
    private _retryCount;
    private _retryInterval;
    private _debouncedFlush;
    private _flushLock;
    /**
     * Timestamp of the last user activity. This lives across sessions.
     */
    private _lastActivity;
    /**
     * Is the integration currently active?
     */
    private _isEnabled;
    /**
     * Paused is a state where:
     * - DOM Recording is not listening at all
     * - Nothing will be added to event buffer (e.g. core SDK events)
     */
    private _isPaused;
    /**
     * Have we attached listeners to the core SDK?
     * Note we have to track this as there is no way to remove instrumentation handlers.
     */
    private _hasInitializedCoreListeners;
    /**
     * Function to stop recording
     */
    private _stopRecording;
    private _context;
    constructor({ options, recordingOptions }: {
        options: ReplayPluginOptions;
        recordingOptions: RecordingOptions;
    });
    /** Get the event context. */
    getContext(): InternalEventContext;
    /** If recording is currently enabled. */
    isEnabled(): boolean;
    /** If recording is currently paused. */
    isPaused(): boolean;
    /** Get the replay integration options. */
    getOptions(): ReplayPluginOptions;
    /**
     * Initializes the plugin.
     *
     * Creates or loads a session, attaches listeners to varying events (DOM,
     * _performanceObserver, Recording, Sentry SDK, etc)
     */
    start(): void;
    /**
     * Start recording.
     *
     * Note that this will cause a new DOM checkout
     */
    startRecording(): void;
    /**
     * Stops the recording, if it was running.
     * Returns true if it was stopped, else false.
     */
    stopRecording(): boolean;
    /**
     * Currently, this needs to be manually called (e.g. for tests). Sentry SDK
     * does not support a teardown
     */
    stop(): void;
    /**
     * Pause some replay functionality. See comments for `_isPaused`.
     * This differs from stop as this only stops DOM recording, it is
     * not as thorough of a shutdown as `stop()`.
     */
    pause(): void;
    /**
     * Resumes recording, see notes for `pause().
     *
     * Note that calling `startRecording()` here will cause a
     * new DOM checkout.`
     */
    resume(): void;
    /** A wrapper to conditionally capture exceptions. */
    handleException(error: unknown): void;
    /** for tests only */
    clearSession(): void;
    /**
     * Loads a session from storage, or creates a new one if it does not exist or
     * is expired.
     */
    loadSession({ expiry }: {
        expiry: number;
    }): void;
    /**
     * Capture some initial state that can change throughout the lifespan of the
     * replay. This is required because otherwise they would be captured at the
     * first flush.
     */
    setInitialState(): void;
    /**
     * Adds listeners to record events for the replay
     */
    addListeners(): void;
    /**
     * Cleans up listeners that were created in `addListeners`
     */
    removeListeners(): void;
    /**
     * We want to batch uploads of replay events. Save events only if
     * `<flushMinDelay>` milliseconds have elapsed since the last event
     * *OR* if `<flushMaxDelay>` milliseconds have elapsed.
     *
     * Accepts a callback to perform side-effects and returns true to stop batch
     * processing and hand back control to caller.
     */
    addUpdate(cb: AddUpdateCallback): void;
    /**
     * Handler for recording events.
     *
     * Adds to event buffer, and has varying flushing behaviors if the event was a checkout.
     */
    handleRecordingEmit: (event: RecordingEvent, isCheckout?: boolean) => void;
    /**
     * Handle when visibility of the page content changes. Opening a new tab will
     * cause the state to change to hidden because of content of current page will
     * be hidden. Likewise, moving a different window to cover the contents of the
     * page will also trigger a change to a hidden state.
     */
    handleVisibilityChange: () => void;
    /**
     * Handle when page is blurred
     */
    handleWindowBlur: () => void;
    /**
     * Handle when page is focused
     */
    handleWindowFocus: () => void;
    /**
     * Handler for Sentry Core SDK events.
     *
     * These events will create breadcrumb-like objects in the recording.
     */
    handleCoreBreadcrumbListener: (type: InstrumentationTypeBreadcrumb) => (handlerData: unknown) => void;
    /**
     * Tasks to run when we consider a page to be hidden (via blurring and/or visibility)
     */
    doChangeToBackgroundTasks(breadcrumb?: Breadcrumb): void;
    /**
     * Tasks to run when we consider a page to be visible (via focus and/or visibility)
     */
    doChangeToForegroundTasks(breadcrumb?: Breadcrumb): void;
    /**
     * Trigger rrweb to take a full snapshot which will cause this plugin to
     * create a new Replay event.
     */
    triggerFullSnapshot(): void;
    /**
     * Update user activity (across session lifespans)
     */
    updateUserActivity(_lastActivity?: number): void;
    /**
     * Updates the session's last activity timestamp
     */
    updateSessionActivity(_lastActivity?: number): void;
    /**
     * Updates the user activity timestamp and resumes recording. This should be
     * called in an event handler for a user action that we consider as the user
     * being "active" (e.g. a mouse click).
     */
    triggerUserActivity(): void;
    /**
     * Helper to create (and buffer) a replay breadcrumb from a core SDK breadcrumb
     */
    createCustomBreadcrumb(breadcrumb: Breadcrumb): void;
    /**
     * Observed performance events are added to `this.performanceEvents`. These
     * are included in the replay event before it is finished and sent to Sentry.
     */
    addPerformanceEntries(): void;
    /**
     * Checks if recording should be stopped due to user inactivity. Otherwise
     * check if session is expired and create a new session if so. Triggers a new
     * full snapshot on new session.
     *
     * Returns true if session is not expired, false otherwise.
     */
    checkAndHandleExpiredSession({ expiry }?: {
        expiry?: number;
    }): boolean | void;
    /**
     * Only flush if `this.recordingMode === 'session'`
     */
    conditionalFlush(): void;
    /**
     * Clear _context
     */
    clearContext(): void;
    /**
     * Return and clear _context
     */
    popEventContext(): PopEventContext;
    /**
     * Flushes replay event buffer to Sentry.
     *
     * Performance events are only added right before flushing - this is
     * due to the buffered performance observer events.
     *
     * Should never be called directly, only by `flush`
     */
    runFlush(): Promise<void>;
    /**
     * Flush recording data to Sentry. Creates a lock so that only a single flush
     * can be active at a time. Do not call this directly.
     */
    flush: () => Promise<void>;
    /**
     *
     * Always flush via `_debouncedFlush` so that we do not have flushes triggered
     * from calling both `flush` and `_debouncedFlush`. Otherwise, there could be
     * cases of mulitple flushes happening closely together.
     */
    flushImmediate(): Promise<void>;
    /**
     * Send replay attachment using `fetch()`
     */
    sendReplayRequest({ events, replayId, segmentId: segment_id, includeReplayStartTimestamp, eventContext, }: SendReplay): Promise<void | TransportMakeRequestResponse>;
    /**
     * Reset the counter of retries for sending replays.
     */
    resetRetries(): void;
    /**
     * Finalize and send the current replay event to Sentry
     */
    sendReplay({ replayId, events, segmentId, includeReplayStartTimestamp, eventContext, }: SendReplay): Promise<unknown>;
    /** Save the session, if it is sticky */
    private _maybeSaveSession;
}
//# sourceMappingURL=replay.d.ts.map