import { _optionalChain } from '@sentry/utils/esm/buildPolyfills';
import { captureException } from '@sentry/core';
import { logger } from '@sentry/utils';
import workerString from './worker/worker.js';

/* eslint-disable @typescript-eslint/no-unsafe-member-access */

/**
 * Create an event buffer for replays.
 */
function createEventBuffer({ useCompression }) {
  // eslint-disable-next-line no-restricted-globals
  if (useCompression && window.Worker) {
    const workerBlob = new Blob([workerString]);
    const workerUrl = URL.createObjectURL(workerBlob);

    try {
      (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && logger.log('[Replay] Using compression worker');
      const worker = new Worker(workerUrl);
      if (worker) {
        return new EventBufferCompressionWorker(worker);
      } else {
        captureException(new Error('Unable to create compression worker'));
      }
    } catch (e) {
      // catch and ignore, fallback to simple event buffer
    }
    (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && logger.log('[Replay] Falling back to simple event buffer');
  }

  (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && logger.log('[Replay] Using simple buffer');
  return new EventBufferArray();
}

class EventBufferArray  {

   constructor() {
    this._events = [];
  }

   get length() {
    return this._events.length;
  }

   destroy() {
    this._events = [];
  }

   addEvent(event, isCheckout) {
    if (isCheckout) {
      this._events = [event];
      return;
    }

    this._events.push(event);
  }

   finish() {
    return new Promise(resolve => {
      // Make a copy of the events array reference and immediately clear the
      // events member so that we do not lose new events while uploading
      // attachment.
      const eventsRet = this._events;
      this._events = [];
      resolve(JSON.stringify(eventsRet));
    });
  }
}

/**
 * Event buffer that uses a web worker to compress events.
 * Exported only for testing.
 */
class EventBufferCompressionWorker  {

   __init() {this._eventBufferItemLength = 0;}
   __init2() {this._id = 0;}

   constructor(worker) {;EventBufferCompressionWorker.prototype.__init.call(this);EventBufferCompressionWorker.prototype.__init2.call(this);
    this._worker = worker;
  }

  /**
   * Note that this may not reflect what is actually in the event buffer. This
   * is only a local count of the buffer size since `addEvent` is async.
   */
   get length() {
    return this._eventBufferItemLength;
  }

  /**
   * Destroy the event buffer.
   */
   destroy() {
    (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && logger.log('[Replay] Destroying compression worker');
    _optionalChain([this, 'access', _ => _._worker, 'optionalAccess', _2 => _2.terminate, 'call', _3 => _3()]);
    this._worker = null;
  }

  /**
   * Add an event to the event buffer.
   */
   async addEvent(event, isCheckout) {
    if (isCheckout) {
      // This event is a checkout, make sure worker buffer is cleared before
      // proceeding.
      await this._postMessage({
        id: this._getAndIncrementId(),
        method: 'init',
        args: [],
      });
    }

    return this._sendEventToWorker(event);
  }

  /**
   * Finish the event buffer and return the compressed data.
   */
   finish() {
    return this._finishRequest(this._getAndIncrementId());
  }

  /**
   * Post message to worker and wait for response before resolving promise.
   */
   _postMessage({ id, method, args }) {
    return new Promise((resolve, reject) => {
      // eslint-disable-next-line @typescript-eslint/explicit-function-return-type
      const listener = ({ data }) => {
        if (data.method !== method) {
          return;
        }

        // There can be multiple listeners for a single method, the id ensures
        // that the response matches the caller.
        if (data.id !== id) {
          return;
        }

        // At this point, we'll always want to remove listener regardless of result status
        _optionalChain([this, 'access', _4 => _4._worker, 'optionalAccess', _5 => _5.removeEventListener, 'call', _6 => _6('message', listener)]);

        if (!data.success) {
          // TODO: Do some error handling, not sure what
          (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && logger.error('[Replay]', data.response);

          reject(new Error('Error in compression worker'));
          return;
        }

        resolve(data.response);
      };

      let stringifiedArgs;
      try {
        stringifiedArgs = JSON.stringify(args);
      } catch (err) {
        (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && logger.error('[Replay] Error when trying to stringify args', err);
        stringifiedArgs = '[]';
      }

      // Note: we can't use `once` option because it's possible it needs to
      // listen to multiple messages
      _optionalChain([this, 'access', _7 => _7._worker, 'optionalAccess', _8 => _8.addEventListener, 'call', _9 => _9('message', listener)]);
      _optionalChain([this, 'access', _10 => _10._worker, 'optionalAccess', _11 => _11.postMessage, 'call', _12 => _12({ id, method, args: stringifiedArgs })]);
    });
  }

  /**
   * Send the event to the worker.
   */
   _sendEventToWorker(event) {
    const promise = this._postMessage({
      id: this._getAndIncrementId(),
      method: 'addEvent',
      args: [event],
    });

    // XXX: See note in `get length()`
    this._eventBufferItemLength++;

    return promise;
  }

  /**
   * Finish the request and return the compressed data from the worker.
   */
   async _finishRequest(id) {
    const promise = this._postMessage({ id, method: 'finish', args: [] });

    // XXX: See note in `get length()`
    this._eventBufferItemLength = 0;

    return promise ;
  }

  /** Get the current ID and increment it for the next call. */
   _getAndIncrementId() {
    return this._id++;
  }
}

export { EventBufferCompressionWorker, createEventBuffer };
//# sourceMappingURL=eventBuffer.js.map
