var {
  _optionalChain
} = require('@sentry/utils/cjs/buildPolyfills');

Object.defineProperty(exports, '__esModule', { value: true });

const core = require('@sentry/core');
const utils = require('@sentry/utils');
require('./node_modules/rrweb/es/rrweb/packages/rrweb/src/entries/all.js');
const constants = require('./constants.js');
const breadcrumbHandler = require('./coreHandlers/breadcrumbHandler.js');
const handleFetch = require('./coreHandlers/handleFetch.js');
const handleGlobalEvent = require('./coreHandlers/handleGlobalEvent.js');
const handleHistory = require('./coreHandlers/handleHistory.js');
const handleXhr = require('./coreHandlers/handleXhr.js');
const performanceObserver = require('./coreHandlers/performanceObserver.js');
const createPerformanceEntry = require('./createPerformanceEntry.js');
const eventBuffer = require('./eventBuffer.js');
const deleteSession = require('./session/deleteSession.js');
const getSession = require('./session/getSession.js');
const saveSession = require('./session/saveSession.js');
const addEvent = require('./util/addEvent.js');
const addMemoryEntry = require('./util/addMemoryEntry.js');
const createBreadcrumb = require('./util/createBreadcrumb.js');
const createPerformanceSpans = require('./util/createPerformanceSpans.js');
const createRecordingData = require('./util/createRecordingData.js');
const createReplayEnvelope = require('./util/createReplayEnvelope.js');
const debounce = require('./util/debounce.js');
const isExpired = require('./util/isExpired.js');
const isSessionExpired = require('./util/isSessionExpired.js');
const monkeyPatchRecordDroppedEvent = require('./util/monkeyPatchRecordDroppedEvent.js');
const prepareReplayEvent = require('./util/prepareReplayEvent.js');
const index = require('./node_modules/rrweb/es/rrweb/packages/rrweb/src/record/index.js');
const types = require('./node_modules/rrweb/es/rrweb/packages/rrweb/src/types.js');

/* eslint-disable max-lines */ // TODO: We might want to split this file up

/**
 * Returns true to return control to calling function, otherwise continue with normal batching
 */

const BASE_RETRY_INTERVAL = 5000;
const MAX_RETRY_COUNT = 3;

/**
 * The main replay container class, which holds all the state and methods for recording and sending replays.
 */
class ReplayContainer  {
   __init() {this.eventBuffer = null;}

  /**
   * List of PerformanceEntry from PerformanceObserver
   */
   __init2() {this.performanceEvents = [];}

  /**
   * Recording can happen in one of two modes:
   * * session: Record the whole session, sending it continuously
   * * error: Always keep the last 60s of recording, and when an error occurs, send it immediately
   */
   __init3() {this.recordingMode = 'session';}

  /**
   * Options to pass to `rrweb.record()`
   */

   __init4() {this._performanceObserver = null;}

   __init5() {this._retryCount = 0;}
   __init6() {this._retryInterval = BASE_RETRY_INTERVAL;}

   __init7() {this._flushLock = null;}

  /**
   * Timestamp of the last user activity. This lives across sessions.
   */
   __init8() {this._lastActivity = new Date().getTime();}

  /**
   * Is the integration currently active?
   */
   __init9() {this._isEnabled = false;}

  /**
   * Paused is a state where:
   * - DOM Recording is not listening at all
   * - Nothing will be added to event buffer (e.g. core SDK events)
   */
   __init10() {this._isPaused = false;}

  /**
   * Have we attached listeners to the core SDK?
   * Note we have to track this as there is no way to remove instrumentation handlers.
   */
   __init11() {this._hasInitializedCoreListeners = false;}

  /**
   * Function to stop recording
   */
   __init12() {this._stopRecording = null;}

   __init13() {this._context = {
    errorIds: new Set(),
    traceIds: new Set(),
    urls: [],
    earliestEvent: null,
    initialTimestamp: new Date().getTime(),
    initialUrl: '',
  };}

  constructor({ options, recordingOptions }) {;ReplayContainer.prototype.__init.call(this);ReplayContainer.prototype.__init2.call(this);ReplayContainer.prototype.__init3.call(this);ReplayContainer.prototype.__init4.call(this);ReplayContainer.prototype.__init5.call(this);ReplayContainer.prototype.__init6.call(this);ReplayContainer.prototype.__init7.call(this);ReplayContainer.prototype.__init8.call(this);ReplayContainer.prototype.__init9.call(this);ReplayContainer.prototype.__init10.call(this);ReplayContainer.prototype.__init11.call(this);ReplayContainer.prototype.__init12.call(this);ReplayContainer.prototype.__init13.call(this);ReplayContainer.prototype.__init14.call(this);ReplayContainer.prototype.__init15.call(this);ReplayContainer.prototype.__init16.call(this);ReplayContainer.prototype.__init17.call(this);ReplayContainer.prototype.__init18.call(this);ReplayContainer.prototype.__init19.call(this);
    this._recordingOptions = recordingOptions;
    this._options = options;

    this._debouncedFlush = debounce.debounce(() => this.flush(), this._options.flushMinDelay, {
      maxWait: this._options.flushMaxDelay,
    });
  }

  /** Get the event context. */
   getContext() {
    return this._context;
  }

  /** If recording is currently enabled. */
   isEnabled() {
    return this._isEnabled;
  }

  /** If recording is currently paused. */
   isPaused() {
    return this._isPaused;
  }

  /** Get the replay integration options. */
   getOptions() {
    return this._options;
  }

  /**
   * Initializes the plugin.
   *
   * Creates or loads a session, attaches listeners to varying events (DOM,
   * _performanceObserver, Recording, Sentry SDK, etc)
   */
  start() {
    this.setInitialState();

    this.loadSession({ expiry: constants.SESSION_IDLE_DURATION });

    // If there is no session, then something bad has happened - can't continue
    if (!this.session) {
      this.handleException(new Error('No session found'));
      return;
    }

    if (!this.session.sampled) {
      // If session was not sampled, then we do not initialize the integration at all.
      return;
    }

    // Modify recording options to checkoutEveryNthSecond if
    // sampling for error replay. This is because we don't know
    // when an error will occur, so we need to keep a buffer of
    // replay events.
    if (this.session.sampled === 'error') {
      this.recordingMode = 'error';
    }

    // setup() is generally called on page load or manually - in both cases we
    // should treat it as an activity
    this.updateSessionActivity();

    this.eventBuffer = eventBuffer.createEventBuffer({
      useCompression: Boolean(this._options.useCompression),
    });

    this.addListeners();

    // Need to set as enabled before we start recording, as `record()` can trigger a flush with a new checkout
    this._isEnabled = true;

    this.startRecording();
  }

  /**
   * Start recording.
   *
   * Note that this will cause a new DOM checkout
   */
  startRecording() {
    try {
      this._stopRecording = index.default({
        ...this._recordingOptions,
        // When running in error sampling mode, we need to overwrite `checkoutEveryNth`
        // Without this, it would record forever, until an error happens, which we don't want
        // instead, we'll always keep the last 60 seconds of replay before an error happened
        ...(this.recordingMode === 'error' && { checkoutEveryNth: 60000 }),
        emit: this.handleRecordingEmit,
      });
    } catch (err) {
      this.handleException(err);
    }
  }

  /**
   * Stops the recording, if it was running.
   * Returns true if it was stopped, else false.
   */
   stopRecording() {
    if (this._stopRecording) {
      this._stopRecording();
      return true;
    }

    return false;
  }

  /**
   * Currently, this needs to be manually called (e.g. for tests). Sentry SDK
   * does not support a teardown
   */
  stop() {
    try {
      (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && utils.logger.log('[Replay] Stopping Replays');
      this._isEnabled = false;
      this.removeListeners();
      _optionalChain([this, 'access', _8 => _8._stopRecording, 'optionalCall', _9 => _9()]);
      _optionalChain([this, 'access', _10 => _10.eventBuffer, 'optionalAccess', _11 => _11.destroy, 'call', _12 => _12()]);
      this.eventBuffer = null;
    } catch (err) {
      this.handleException(err);
    }
  }

  /**
   * Pause some replay functionality. See comments for `_isPaused`.
   * This differs from stop as this only stops DOM recording, it is
   * not as thorough of a shutdown as `stop()`.
   */
  pause() {
    this._isPaused = true;
    try {
      if (this._stopRecording) {
        this._stopRecording();
        this._stopRecording = undefined;
      }
    } catch (err) {
      this.handleException(err);
    }
  }

  /**
   * Resumes recording, see notes for `pause().
   *
   * Note that calling `startRecording()` here will cause a
   * new DOM checkout.`
   */
  resume() {
    this._isPaused = false;
    this.startRecording();
  }

  /** A wrapper to conditionally capture exceptions. */
  handleException(error) {
    (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && utils.logger.error('[Replay]', error);

    if ((typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && this._options._experiments && this._options._experiments.captureExceptions) {
      core.captureException(error);
    }
  }

  /** for tests only */
  clearSession() {
    try {
      deleteSession.deleteSession();
      this.session = undefined;
    } catch (err) {
      this.handleException(err);
    }
  }

  /**
   * Loads a session from storage, or creates a new one if it does not exist or
   * is expired.
   */
  loadSession({ expiry }) {
    const { type, session } = getSession.getSession({
      expiry,
      stickySession: Boolean(this._options.stickySession),
      currentSession: this.session,
      sessionSampleRate: this._options.sessionSampleRate,
      errorSampleRate: this._options.errorSampleRate,
    });

    // If session was newly created (i.e. was not loaded from storage), then
    // enable flag to create the root replay
    if (type === 'new') {
      this.setInitialState();
    }

    if (session.id !== _optionalChain([this, 'access', _13 => _13.session, 'optionalAccess', _14 => _14.id])) {
      session.previousSessionId = _optionalChain([this, 'access', _15 => _15.session, 'optionalAccess', _16 => _16.id]);
    }

    this.session = session;
  }

  /**
   * Capture some initial state that can change throughout the lifespan of the
   * replay. This is required because otherwise they would be captured at the
   * first flush.
   */
  setInitialState() {
    const urlPath = `${constants.WINDOW.location.pathname}${constants.WINDOW.location.hash}${constants.WINDOW.location.search}`;
    const url = `${constants.WINDOW.location.origin}${urlPath}`;

    this.performanceEvents = [];

    // Reset _context as well
    this.clearContext();

    this._context.initialUrl = url;
    this._context.initialTimestamp = new Date().getTime();
    this._context.urls.push(url);
  }

  /**
   * Adds listeners to record events for the replay
   */
  addListeners() {
    try {
      constants.WINDOW.document.addEventListener('visibilitychange', this.handleVisibilityChange);
      constants.WINDOW.addEventListener('blur', this.handleWindowBlur);
      constants.WINDOW.addEventListener('focus', this.handleWindowFocus);

      // We need to filter out dropped events captured by `addGlobalEventProcessor(this.handleGlobalEvent)` below
      monkeyPatchRecordDroppedEvent.overwriteRecordDroppedEvent(this._context.errorIds);

      // There is no way to remove these listeners, so ensure they are only added once
      if (!this._hasInitializedCoreListeners) {
        // Listeners from core SDK //
        const scope = core.getCurrentHub().getScope();
        _optionalChain([scope, 'optionalAccess', _17 => _17.addScopeListener, 'call', _18 => _18(this.handleCoreBreadcrumbListener('scope'))]);
        utils.addInstrumentationHandler('dom', this.handleCoreBreadcrumbListener('dom'));
        utils.addInstrumentationHandler('fetch', handleFetch.handleFetchSpanListener(this));
        utils.addInstrumentationHandler('xhr', handleXhr.handleXhrSpanListener(this));
        utils.addInstrumentationHandler('history', handleHistory.handleHistorySpanListener(this));

        // Tag all (non replay) events that get sent to Sentry with the current
        // replay ID so that we can reference them later in the UI
        core.addGlobalEventProcessor(handleGlobalEvent.handleGlobalEventListener(this));

        this._hasInitializedCoreListeners = true;
      }
    } catch (err) {
      this.handleException(err);
    }

    // _performanceObserver //
    if (!('_performanceObserver' in constants.WINDOW)) {
      return;
    }

    this._performanceObserver = performanceObserver.setupPerformanceObserver(this);
  }

  /**
   * Cleans up listeners that were created in `addListeners`
   */
  removeListeners() {
    try {
      constants.WINDOW.document.removeEventListener('visibilitychange', this.handleVisibilityChange);

      constants.WINDOW.removeEventListener('blur', this.handleWindowBlur);
      constants.WINDOW.removeEventListener('focus', this.handleWindowFocus);

      monkeyPatchRecordDroppedEvent.restoreRecordDroppedEvent();

      if (this._performanceObserver) {
        this._performanceObserver.disconnect();
        this._performanceObserver = null;
      }
    } catch (err) {
      this.handleException(err);
    }
  }

  /**
   * We want to batch uploads of replay events. Save events only if
   * `<flushMinDelay>` milliseconds have elapsed since the last event
   * *OR* if `<flushMaxDelay>` milliseconds have elapsed.
   *
   * Accepts a callback to perform side-effects and returns true to stop batch
   * processing and hand back control to caller.
   */
  addUpdate(cb) {
    // We need to always run `cb` (e.g. in the case of `this.recordingMode == 'error'`)
    const cbResult = _optionalChain([cb, 'optionalCall', _19 => _19()]);

    // If this option is turned on then we will only want to call `flush`
    // explicitly
    if (this.recordingMode === 'error') {
      return;
    }

    // If callback is true, we do not want to continue with flushing -- the
    // caller will need to handle it.
    if (cbResult === true) {
      return;
    }

    // addUpdate is called quite frequently - use _debouncedFlush so that it
    // respects the flush delays and does not flush immediately
    this._debouncedFlush();
  }

  /**
   * Handler for recording events.
   *
   * Adds to event buffer, and has varying flushing behaviors if the event was a checkout.
   */
  __init14() {this.handleRecordingEmit = (
    event,
    isCheckout,
  ) => {
    // If this is false, it means session is expired, create and a new session and wait for checkout
    if (!this.checkAndHandleExpiredSession()) {
      (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && utils.logger.error('[Replay] Received replay event after session expired.');

      return;
    }

    this.addUpdate(() => {
      // The session is always started immediately on pageload/init, but for
      // error-only replays, it should reflect the most recent checkout
      // when an error occurs. Clear any state that happens before this current
      // checkout. This needs to happen before `addEvent()` which updates state
      // dependent on this reset.
      if (this.recordingMode === 'error' && event.type === 2) {
        this.setInitialState();
      }

      // We need to clear existing events on a checkout, otherwise they are
      // incremental event updates and should be appended
      addEvent.addEvent(this, event, isCheckout);

      // Different behavior for full snapshots (type=2), ignore other event types
      // See https://github.com/rrweb-io/rrweb/blob/d8f9290ca496712aa1e7d472549480c4e7876594/packages/rrweb/src/types.ts#L16
      if (event.type !== 2) {
        return false;
      }

      // If there is a previousSessionId after a full snapshot occurs, then
      // the replay session was started due to session expiration. The new session
      // is started before triggering a new checkout and contains the id
      // of the previous session. Do not immediately flush in this case
      // to avoid capturing only the checkout and instead the replay will
      // be captured if they perform any follow-up actions.
      if (_optionalChain([this, 'access', _20 => _20.session, 'optionalAccess', _21 => _21.previousSessionId])) {
        return true;
      }

      // See note above re: session start needs to reflect the most recent
      // checkout.
      if (this.recordingMode === 'error' && this.session && this._context.earliestEvent) {
        this.session.started = this._context.earliestEvent;
        this._maybeSaveSession();
      }

      // Flush immediately so that we do not miss the first segment, otherwise
      // it can prevent loading on the UI. This will cause an increase in short
      // replays (e.g. opening and closing a tab quickly), but these can be
      // filtered on the UI.
      if (this.recordingMode === 'session') {
        void this.flushImmediate();
      }

      return true;
    });
  };}

  /**
   * Handle when visibility of the page content changes. Opening a new tab will
   * cause the state to change to hidden because of content of current page will
   * be hidden. Likewise, moving a different window to cover the contents of the
   * page will also trigger a change to a hidden state.
   */
  __init15() {this.handleVisibilityChange = () => {
    if (constants.WINDOW.document.visibilityState === 'visible') {
      this.doChangeToForegroundTasks();
    } else {
      this.doChangeToBackgroundTasks();
    }
  };}

  /**
   * Handle when page is blurred
   */
  __init16() {this.handleWindowBlur = () => {
    const breadcrumb = createBreadcrumb.createBreadcrumb({
      category: 'ui.blur',
    });

    // Do not count blur as a user action -- it's part of the process of them
    // leaving the page
    this.doChangeToBackgroundTasks(breadcrumb);
  };}

  /**
   * Handle when page is focused
   */
  __init17() {this.handleWindowFocus = () => {
    const breadcrumb = createBreadcrumb.createBreadcrumb({
      category: 'ui.focus',
    });

    // Do not count focus as a user action -- instead wait until they focus and
    // interactive with page
    this.doChangeToForegroundTasks(breadcrumb);
  };}

  /**
   * Handler for Sentry Core SDK events.
   *
   * These events will create breadcrumb-like objects in the recording.
   */
  __init18() {this.handleCoreBreadcrumbListener =
    (type) =>
    (handlerData) => {
      if (!this._isEnabled) {
        return;
      }

      const result = breadcrumbHandler.breadcrumbHandler(type, handlerData);

      if (result === null) {
        return;
      }

      if (result.category === 'sentry.transaction') {
        return;
      }

      if (result.category === 'ui.click') {
        this.triggerUserActivity();
      } else {
        this.checkAndHandleExpiredSession();
      }

      this.addUpdate(() => {
        addEvent.addEvent(this, {
          type: types.EventType.Custom,
          // TODO: We were converting from ms to seconds for breadcrumbs, spans,
          // but maybe we should just keep them as milliseconds
          timestamp: (result.timestamp || 0) * 1000,
          data: {
            tag: 'breadcrumb',
            payload: result,
          },
        });

        // Do not flush after console log messages
        return result.category === 'console';
      });
    };}

  /**
   * Tasks to run when we consider a page to be hidden (via blurring and/or visibility)
   */
  doChangeToBackgroundTasks(breadcrumb) {
    if (!this.session) {
      return;
    }

    const expired = isSessionExpired.isSessionExpired(this.session, constants.VISIBILITY_CHANGE_TIMEOUT);

    if (breadcrumb && !expired) {
      this.createCustomBreadcrumb(breadcrumb);
    }

    // Send replay when the page/tab becomes hidden. There is no reason to send
    // replay if it becomes visible, since no actions we care about were done
    // while it was hidden
    this.conditionalFlush();
  }

  /**
   * Tasks to run when we consider a page to be visible (via focus and/or visibility)
   */
  doChangeToForegroundTasks(breadcrumb) {
    if (!this.session) {
      return;
    }

    const isSessionActive = this.checkAndHandleExpiredSession({
      expiry: constants.VISIBILITY_CHANGE_TIMEOUT,
    });

    if (!isSessionActive) {
      // If the user has come back to the page within VISIBILITY_CHANGE_TIMEOUT
      // ms, we will re-use the existing session, otherwise create a new
      // session
      (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && utils.logger.log('[Replay] Document has become active, but session has expired');
      return;
    }

    if (breadcrumb) {
      this.createCustomBreadcrumb(breadcrumb);
    }
  }

  /**
   * Trigger rrweb to take a full snapshot which will cause this plugin to
   * create a new Replay event.
   */
  triggerFullSnapshot() {
    (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && utils.logger.log('[Replay] Taking full rrweb snapshot');
    index.default.takeFullSnapshot(true);
  }

  /**
   * Update user activity (across session lifespans)
   */
  updateUserActivity(_lastActivity = new Date().getTime()) {
    this._lastActivity = _lastActivity;
  }

  /**
   * Updates the session's last activity timestamp
   */
  updateSessionActivity(_lastActivity = new Date().getTime()) {
    if (this.session) {
      this.session.lastActivity = _lastActivity;
      this._maybeSaveSession();
    }
  }

  /**
   * Updates the user activity timestamp and resumes recording. This should be
   * called in an event handler for a user action that we consider as the user
   * being "active" (e.g. a mouse click).
   */
  triggerUserActivity() {
    this.updateUserActivity();

    // This case means that recording was once stopped due to inactivity.
    // Ensure that recording is resumed.
    if (!this._stopRecording) {
      // Create a new session, otherwise when the user action is flushed, it
      // will get rejected due to an expired session.
      this.loadSession({ expiry: constants.SESSION_IDLE_DURATION });

      // Note: This will cause a new DOM checkout
      this.resume();
      return;
    }

    // Otherwise... recording was never suspended, continue as normalish
    this.checkAndHandleExpiredSession();

    this.updateSessionActivity();
  }

  /**
   * Helper to create (and buffer) a replay breadcrumb from a core SDK breadcrumb
   */
  createCustomBreadcrumb(breadcrumb) {
    this.addUpdate(() => {
      addEvent.addEvent(this, {
        type: types.EventType.Custom,
        timestamp: breadcrumb.timestamp || 0,
        data: {
          tag: 'breadcrumb',
          payload: breadcrumb,
        },
      });
    });
  }

  /**
   * Observed performance events are added to `this.performanceEvents`. These
   * are included in the replay event before it is finished and sent to Sentry.
   */
  addPerformanceEntries() {
    // Copy and reset entries before processing
    const entries = [...this.performanceEvents];
    this.performanceEvents = [];

    createPerformanceSpans.createPerformanceSpans(this, createPerformanceEntry.createPerformanceEntries(entries));
  }

  /**
   * Checks if recording should be stopped due to user inactivity. Otherwise
   * check if session is expired and create a new session if so. Triggers a new
   * full snapshot on new session.
   *
   * Returns true if session is not expired, false otherwise.
   */
  checkAndHandleExpiredSession({ expiry = constants.SESSION_IDLE_DURATION } = {}) {
    const oldSessionId = _optionalChain([this, 'access', _22 => _22.session, 'optionalAccess', _23 => _23.id]);

    // Prevent starting a new session if the last user activity is older than
    // MAX_SESSION_LIFE. Otherwise non-user activity can trigger a new
    // session+recording. This creates noisy replays that do not have much
    // content in them.
    if (this._lastActivity && isExpired.isExpired(this._lastActivity, constants.MAX_SESSION_LIFE)) {
      // Pause recording
      this.pause();
      return;
    }

    // --- There is recent user activity --- //
    // This will create a new session if expired, based on expiry length
    this.loadSession({ expiry });

    // Session was expired if session ids do not match
    const expired = oldSessionId !== _optionalChain([this, 'access', _24 => _24.session, 'optionalAccess', _25 => _25.id]);

    if (!expired) {
      return true;
    }

    // Session is expired, trigger a full snapshot (which will create a new session)
    this.triggerFullSnapshot();

    return false;
  }

  /**
   * Only flush if `this.recordingMode === 'session'`
   */
  conditionalFlush() {
    if (this.recordingMode === 'error') {
      return;
    }

    void this.flushImmediate();
  }

  /**
   * Clear _context
   */
  clearContext() {
    // XXX: `initialTimestamp` and `initialUrl` do not get cleared
    this._context.errorIds.clear();
    this._context.traceIds.clear();
    this._context.urls = [];
    this._context.earliestEvent = null;
  }

  /**
   * Return and clear _context
   */
  popEventContext() {
    if (this._context.earliestEvent && this._context.earliestEvent < this._context.initialTimestamp) {
      this._context.initialTimestamp = this._context.earliestEvent;
    }

    const _context = {
      initialTimestamp: this._context.initialTimestamp,
      initialUrl: this._context.initialUrl,
      errorIds: Array.from(this._context.errorIds).filter(Boolean),
      traceIds: Array.from(this._context.traceIds).filter(Boolean),
      urls: this._context.urls,
    };

    this.clearContext();

    return _context;
  }

  /**
   * Flushes replay event buffer to Sentry.
   *
   * Performance events are only added right before flushing - this is
   * due to the buffered performance observer events.
   *
   * Should never be called directly, only by `flush`
   */
  async runFlush() {
    if (!this.session) {
      (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && utils.logger.error('[Replay] No session found to flush.');
      return;
    }

    await this.addPerformanceEntries();

    if (!_optionalChain([this, 'access', _26 => _26.eventBuffer, 'optionalAccess', _27 => _27.length])) {
      return;
    }

    // Only attach memory event if eventBuffer is not empty
    await addMemoryEntry.addMemoryEntry(this);

    try {
      // Note this empties the event buffer regardless of outcome of sending replay
      const recordingData = await this.eventBuffer.finish();

      // NOTE: Copy values from instance members, as it's possible they could
      // change before the flush finishes.
      const replayId = this.session.id;
      const eventContext = this.popEventContext();
      // Always increment segmentId regardless of outcome of sending replay
      const segmentId = this.session.segmentId++;
      this._maybeSaveSession();

      await this.sendReplay({
        replayId,
        events: recordingData,
        segmentId,
        includeReplayStartTimestamp: segmentId === 0,
        eventContext,
      });
    } catch (err) {
      this.handleException(err);
    }
  }

  /**
   * Flush recording data to Sentry. Creates a lock so that only a single flush
   * can be active at a time. Do not call this directly.
   */
  __init19() {this.flush = async () => {
    if (!this._isEnabled) {
      // This is just a precaution, there should be no listeners that would
      // cause a flush.
      return;
    }

    if (!this.checkAndHandleExpiredSession()) {
      (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && utils.logger.error('[Replay] Attempting to finish replay event after session expired.');
      return;
    }

    if (!_optionalChain([this, 'access', _28 => _28.session, 'optionalAccess', _29 => _29.id])) {
      (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && utils.logger.error('[Replay] No session found to flush.');
      return;
    }

    // A flush is about to happen, cancel any queued flushes
    _optionalChain([this, 'access', _30 => _30._debouncedFlush, 'optionalAccess', _31 => _31.cancel, 'call', _32 => _32()]);

    // this._flushLock acts as a lock so that future calls to `flush()`
    // will be blocked until this promise resolves
    if (!this._flushLock) {
      this._flushLock = this.runFlush();
      await this._flushLock;
      this._flushLock = null;
      return;
    }

    // Wait for previous flush to finish, then call the debounced `flush()`.
    // It's possible there are other flush requests queued and waiting for it
    // to resolve. We want to reduce all outstanding requests (as well as any
    // new flush requests that occur within a second of the locked flush
    // completing) into a single flush.

    try {
      await this._flushLock;
    } catch (err) {
      (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && utils.logger.error(err);
    } finally {
      this._debouncedFlush();
    }
  };}

  /**
   *
   * Always flush via `_debouncedFlush` so that we do not have flushes triggered
   * from calling both `flush` and `_debouncedFlush`. Otherwise, there could be
   * cases of mulitple flushes happening closely together.
   */
  flushImmediate() {
    this._debouncedFlush();
    // `.flush` is provided by the debounced function, analogously to lodash.debounce
    return this._debouncedFlush.flush() ;
  }

  /**
   * Send replay attachment using `fetch()`
   */
  async sendReplayRequest({
    events,
    replayId,
    segmentId: segment_id,
    includeReplayStartTimestamp,
    eventContext,
  }) {
    const recordingData = createRecordingData.createRecordingData({
      events,
      headers: {
        segment_id,
      },
    });

    const { urls, errorIds, traceIds, initialTimestamp } = eventContext;

    const currentTimestamp = new Date().getTime();

    const hub = core.getCurrentHub();
    const client = hub.getClient();
    const scope = hub.getScope();
    const transport = client && client.getTransport();
    const dsn = _optionalChain([client, 'optionalAccess', _33 => _33.getDsn, 'call', _34 => _34()]);

    if (!client || !scope || !transport || !dsn || !this.session || !this.session.sampled) {
      return;
    }

    const baseEvent = {
      // @ts-ignore private api
      type: constants.REPLAY_EVENT_NAME,
      ...(includeReplayStartTimestamp ? { replay_start_timestamp: initialTimestamp / 1000 } : {}),
      timestamp: currentTimestamp / 1000,
      error_ids: errorIds,
      trace_ids: traceIds,
      urls,
      replay_id: replayId,
      segment_id,
      replay_type: this.session.sampled,
    };

    const replayEvent = await prepareReplayEvent.prepareReplayEvent({ scope, client, replayId, event: baseEvent });

    if (!replayEvent) {
      // Taken from baseclient's `_processEvent` method, where this is handled for errors/transactions
      client.recordDroppedEvent('event_processor', 'replay_event', baseEvent);
      (typeof __SENTRY_DEBUG__ === 'undefined' || __SENTRY_DEBUG__) && utils.logger.log('An event processor returned `null`, will not send event.');
      return;
    }

    replayEvent.tags = {
      ...replayEvent.tags,
      sessionSampleRate: this._options.sessionSampleRate,
      errorSampleRate: this._options.errorSampleRate,
    };

    /*
    For reference, the fully built event looks something like this:
    {
        "type": "replay_event",
        "timestamp": 1670837008.634,
        "error_ids": [
            "errorId"
        ],
        "trace_ids": [
            "traceId"
        ],
        "urls": [
            "https://example.com"
        ],
        "replay_id": "eventId",
        "segment_id": 3,
        "replay_type": "error",
        "platform": "javascript",
        "event_id": "eventId",
        "environment": "production",
        "sdk": {
            "integrations": [
                "BrowserTracing",
                "Replay"
            ],
            "name": "sentry.javascript.browser",
            "version": "7.25.0"
        },
        "sdkProcessingMetadata": {},
        "tags": {
            "sessionSampleRate": 1,
            "errorSampleRate": 0,
        }
    }
    */

    const envelope = createReplayEnvelope.createReplayEnvelope(replayEvent, recordingData, dsn, client.getOptions().tunnel);

    try {
      return await transport.send(envelope);
    } catch (e) {
      throw new Error(constants.UNABLE_TO_SEND_REPLAY);
    }
  }

  /**
   * Reset the counter of retries for sending replays.
   */
  resetRetries() {
    this._retryCount = 0;
    this._retryInterval = BASE_RETRY_INTERVAL;
  }

  /**
   * Finalize and send the current replay event to Sentry
   */
  async sendReplay({
    replayId,
    events,
    segmentId,
    includeReplayStartTimestamp,
    eventContext,
  }) {
    // short circuit if there's no events to upload (this shouldn't happen as runFlush makes this check)
    if (!events.length) {
      return;
    }

    try {
      await this.sendReplayRequest({
        events,
        replayId,
        segmentId,
        includeReplayStartTimestamp,
        eventContext,
      });
      this.resetRetries();
      return true;
    } catch (err) {
      // Capture error for every failed replay
      core.setContext('Replays', {
        _retryCount: this._retryCount,
      });
      this.handleException(err);

      // If an error happened here, it's likely that uploading the attachment
      // failed, we'll can retry with the same events payload
      if (this._retryCount >= MAX_RETRY_COUNT) {
        throw new Error(`${constants.UNABLE_TO_SEND_REPLAY} - max retries exceeded`);
      }

      this._retryCount = this._retryCount + 1;
      // will retry in intervals of 5, 10, 30
      this._retryInterval = this._retryCount * this._retryInterval;

      return await new Promise((resolve, reject) => {
        setTimeout(async () => {
          try {
            await this.sendReplay({
              replayId,
              events,
              segmentId,
              includeReplayStartTimestamp,
              eventContext,
            });
            resolve(true);
          } catch (err) {
            reject(err);
          }
        }, this._retryInterval);
      });
    }
  }

  /** Save the session, if it is sticky */
   _maybeSaveSession() {
    if (this.session && this._options.stickySession) {
      saveSession.saveSession(this.session);
    }
  }
}

exports.ReplayContainer = ReplayContainer;
//# sourceMappingURL=replay.js.map
