Object.defineProperty(exports, "__esModule", { value: true });
exports.makeElectronOfflineTransport = void 0;
const tslib_1 = require("tslib");
const core_1 = require("@sentry/core");
const utils_1 = require("@sentry/utils");
const electron_1 = require("electron");
const path_1 = require("path");
const fs_1 = require("../fs");
const electron_net_1 = require("./electron-net");
const queue_1 = require("./queue");
const START_DELAY = 5000;
const MAX_DELAY = 2000000000;
/** Returns true is there's a chance we're online */
function maybeOnline() {
    return !('online' in electron_1.net) || electron_1.net.online === true;
}
function defaultBeforeSend(_) {
    return maybeOnline() ? 'send' : 'queue';
}
function isRateLimited(result) {
    return !!(result.headers && 'x-sentry-rate-limits' in result.headers);
}
/**
 * Creates a Transport that uses Electrons net module to send events to Sentry. When they fail to send they are
 * persisted to disk and sent later
 */
function makeElectronOfflineTransport(options) {
    const netMakeRequest = (0, electron_net_1.createElectronNetRequestExecutor)(options.url, options.headers || {});
    const queue = new queue_1.PersistedRequestQueue((0, path_1.join)(fs_1.sentryCachePath, 'queue'), options.maxQueueAgeDays, options.maxQueueCount);
    const beforeSend = options.beforeSend || defaultBeforeSend;
    let retryDelay = START_DELAY;
    let lastQueueLength = -1;
    function queueLengthChanged(queuedEvents) {
        if (options.queuedLengthChanged && queuedEvents !== lastQueueLength) {
            lastQueueLength = queuedEvents;
            options.queuedLengthChanged(queuedEvents);
        }
    }
    function flushQueue() {
        queue
            .pop()
            .then((found) => {
            if (found) {
                // We have pendingCount plus found.request
                queueLengthChanged(found.pendingCount + 1);
                utils_1.logger.log('Found a request in the queue');
                makeRequest(found.request).catch((e) => utils_1.logger.error(e));
            }
            else {
                queueLengthChanged(0);
            }
        })
            .catch((e) => utils_1.logger.error(e));
    }
    function queueRequest(request) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            utils_1.logger.log('Queuing request');
            queueLengthChanged(yield queue.add(request));
            setTimeout(() => {
                flushQueue();
            }, retryDelay);
            retryDelay *= 3;
            // If the delay is bigger than 2^31 (max signed 32-bit int), setTimeout throws
            // an error and falls back to 1 which can cause a huge number of requests.
            if (retryDelay > MAX_DELAY) {
                retryDelay = MAX_DELAY;
            }
            return {};
        });
    }
    function makeRequest(request) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            let action = beforeSend(request);
            if (action instanceof Promise) {
                action = yield action;
            }
            if (action === 'send') {
                try {
                    const result = yield netMakeRequest(request);
                    if (!isRateLimited(result)) {
                        utils_1.logger.log('Successfully sent');
                        // Reset the retry delay
                        retryDelay = START_DELAY;
                        // We were successful so check the queue
                        flushQueue();
                        return result;
                    }
                    else {
                        utils_1.logger.log('Rate limited', result.headers);
                    }
                }
                catch (error) {
                    utils_1.logger.log('Error sending:', error);
                }
                action = 'queue';
            }
            if (action == 'queue') {
                return yield queueRequest(request);
            }
            utils_1.logger.log('Dropping request');
            return {};
        });
    }
    flushQueue();
    return (0, core_1.createTransport)(options, makeRequest);
}
exports.makeElectronOfflineTransport = makeElectronOfflineTransport;
//# sourceMappingURL=electron-offline-net.js.map