Object.defineProperty(exports, "__esModule", { value: true });
exports.BufferedWriteStore = exports.Store = void 0;
const tslib_1 = require("tslib");
const utils_1 = require("@sentry/utils");
const path_1 = require("path");
const mutex_1 = require("../common/mutex");
const fs_1 = require("./fs");
const dateFormat = /^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.*\d{0,10}Z$/;
/** JSON revive function to enable de-serialization of Date objects */
function dateReviver(_, value) {
    if (typeof value === 'string' && dateFormat.test(value)) {
        return new Date(value);
    }
    return value;
}
/**
 * Stores data serialized to a JSON file.
 */
class Store {
    /**
     * Creates a new store.
     *
     * @param path A unique filename to store this data.
     * @param id A unique filename to store this data.
     * @param initial An initial value to initialize data with.
     */
    constructor(path, id, initial) {
        /** A mutex to ensure that there aren't races while reading and writing files */
        this._lock = new mutex_1.Mutex();
        this._path = (0, path_1.join)(path, `${id}.json`);
        this._initial = initial;
    }
    /**
     * Updates data by replacing it with the given value.
     * @param data New data to replace the previous one.
     */
    set(data) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this._lock.runExclusive(() => tslib_1.__awaiter(this, void 0, void 0, function* () {
                this._data = data;
                try {
                    if (data === undefined) {
                        try {
                            yield (0, fs_1.unlinkAsync)(this._path);
                        }
                        catch (_) {
                            //
                        }
                    }
                    else {
                        yield (0, fs_1.mkdirp)((0, path_1.dirname)(this._path));
                        yield (0, fs_1.writeFileAsync)(this._path, JSON.stringify(data));
                    }
                }
                catch (e) {
                    utils_1.logger.warn('Failed to write to store', e);
                    // This usually fails due to anti virus scanners, issues in the file
                    // system, or problems with network drives. We cannot fix or handle this
                    // issue and must resume gracefully. Thus, we have to ignore this error.
                }
            }));
        });
    }
    /**
     * Returns the current data.
     *
     * When invoked for the first time, it will try to load previously stored data
     * from disk. If the file does not exist, the initial value provided to the
     * constructor is used.
     */
    get() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return yield this._lock.runExclusive(() => tslib_1.__awaiter(this, void 0, void 0, function* () {
                if (this._data === undefined) {
                    try {
                        this._data = JSON.parse(yield (0, fs_1.readFileAsync)(this._path, 'utf8'), dateReviver);
                    }
                    catch (e) {
                        this._data = this._initial;
                    }
                }
                return this._data;
            }));
        });
    }
    /**
     * Updates data by passing it through the given function.
     * @param fn A function receiving the current data and returning new one.
     */
    update(fn) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.set(fn(yield this.get()));
        });
    }
    /** Returns store to its initial state */
    clear() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            yield this.set(this._initial);
        });
    }
    /** Gets the Date that the file was last modified */
    getModifiedDate() {
        var _a;
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                return (_a = (yield (0, fs_1.statAsync)(this._path))) === null || _a === void 0 ? void 0 : _a.mtime;
            }
            catch (_) {
                return undefined;
            }
        });
    }
}
exports.Store = Store;
/**
 * Extends Store to throttle writes.
 */
class BufferedWriteStore extends Store {
    /**
     * Creates a new ThrottledStore.
     *
     * @param path A unique filename to store this data.
     * @param id A unique filename to store this data.
     * @param initial An initial value to initialize data with.
     * @param throttleTime The minimum time between writes
     */
    constructor(path, id, initial, throttleTime = 500) {
        super(path, id, initial);
        this._throttleTime = throttleTime;
    }
    /** @inheritdoc */
    set(data) {
        var _a;
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            this._data = data;
            this._pendingWrite = {
                // We overwrite the data for the pending write so that the latest data is written in the next flush
                data,
                // If there is already a pending timeout, we keep that rather than starting the timeout again
                timeout: ((_a = this._pendingWrite) === null || _a === void 0 ? void 0 : _a.timeout) || setTimeout(() => this._writePending(), this._throttleTime),
            };
        });
    }
    /** Writes the pending write to disk */
    _writePending() {
        if (this._pendingWrite) {
            const data = this._pendingWrite.data;
            // Clear the pending write immediately so that subsequent writes can be queued
            this._pendingWrite = undefined;
            void super.set(data);
        }
    }
}
exports.BufferedWriteStore = BufferedWriteStore;
//# sourceMappingURL=store.js.map