Object.defineProperty(exports, "__esModule", { value: true });
exports.sessionCrashed = exports.checkPreviousSession = exports.unreportedDuringLastSession = exports.endSession = exports.startSession = void 0;
const tslib_1 = require("tslib");
const core_1 = require("@sentry/core");
const node_1 = require("@sentry/node");
const utils_1 = require("@sentry/utils");
const fs_1 = require("./fs");
const store_1 = require("./store");
const PERSIST_INTERVAL_MS = 60000;
/** Stores the app session in case of termination due to main process crash or app killed */
const sessionStore = new store_1.Store(fs_1.sentryCachePath, 'session', undefined);
/** Previous session that did not exit cleanly */
let previousSession = sessionStore.get();
let persistTimer;
/** Starts a session */
function startSession() {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const hub = (0, core_1.getCurrentHub)();
        yield sessionStore.set(hub.startSession());
        // Every PERSIST_INTERVAL, write the session to disk
        persistTimer = setInterval(() => tslib_1.__awaiter(this, void 0, void 0, function* () {
            var _a;
            const currentSession = (_a = hub.getScope()) === null || _a === void 0 ? void 0 : _a.getSession();
            // Only bother saving if it hasn't already ended
            if (currentSession && currentSession.status === 'ok') {
                yield sessionStore.set(currentSession);
            }
        }), PERSIST_INTERVAL_MS);
    });
}
exports.startSession = startSession;
/** Cleanly ends a session */
function endSession() {
    var _a;
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        // Once the session had ended there is no point persisting it
        if (persistTimer) {
            clearInterval(persistTimer);
        }
        const hub = (0, core_1.getCurrentHub)();
        const session = (_a = hub.getScope()) === null || _a === void 0 ? void 0 : _a.getSession();
        if (session) {
            if (session.status === 'ok') {
                utils_1.logger.log('Ending session');
                hub.endSession();
            }
            else {
                utils_1.logger.log('Session was already ended');
            }
        }
        else {
            utils_1.logger.log('No session');
        }
        yield sessionStore.clear();
        yield (0, node_1.flush)();
    });
}
exports.endSession = endSession;
/** Determines if a Date is likely to have occurred in the previous uncompleted session */
function unreportedDuringLastSession(crashDate) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (!crashDate) {
            return false;
        }
        const previousSessionModified = yield sessionStore.getModifiedDate();
        // There is no previous session
        if (previousSessionModified == undefined) {
            return false;
        }
        const previousSessionModifiedTime = previousSessionModified.getTime();
        const crashTime = crashDate.getTime();
        // Session could have run until modified time + persist interval
        const prevSessionEnd = previousSessionModifiedTime + PERSIST_INTERVAL_MS;
        // Event cannot have occurred before last persist time, We add a 2 second overlap to be sure
        const lastPersist = previousSessionModifiedTime - 2000;
        // If the crash occurred between the last persist and estimated end of session
        return crashTime > lastPersist && crashTime < prevSessionEnd;
    });
}
exports.unreportedDuringLastSession = unreportedDuringLastSession;
/** Checks if the previous session needs sending as crashed or abnormal  */
function checkPreviousSession(crashed) {
    var _a, _b;
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const client = (0, core_1.getCurrentHub)().getClient();
        const previous = yield previousSession;
        if (previous && client) {
            // Ignore if the previous session is already ended
            if (previous.status !== 'ok') {
                previousSession = undefined;
                return;
            }
            const status = crashed ? 'crashed' : 'abnormal';
            utils_1.logger.log(`Found previous ${status} session`);
            const sesh = (0, core_1.makeSession)(previous);
            (0, core_1.updateSession)(sesh, {
                status,
                errors: (sesh.errors || 0) + 1,
                release: (_a = previous.attrs) === null || _a === void 0 ? void 0 : _a.release,
                environment: (_b = previous.attrs) === null || _b === void 0 ? void 0 : _b.environment,
            });
            yield client.sendSession(sesh);
            previousSession = undefined;
        }
    });
}
exports.checkPreviousSession = checkPreviousSession;
/** Sets the current session as crashed */
function sessionCrashed() {
    var _a;
    // stop persisting session
    if (persistTimer) {
        clearInterval(persistTimer);
    }
    utils_1.logger.log('Session Crashed');
    const hub = (0, core_1.getCurrentHub)();
    const session = (_a = hub.getScope()) === null || _a === void 0 ? void 0 : _a.getSession();
    if (!session) {
        utils_1.logger.log('No session to update');
        return;
    }
    if (session.status === 'ok') {
        utils_1.logger.log('Setting session as crashed');
        (0, core_1.updateSession)(session, { status: 'crashed', errors: (session.errors += 1) });
    }
    else {
        utils_1.logger.log('Session already ended');
    }
    hub.captureSession();
}
exports.sessionCrashed = sessionCrashed;
//# sourceMappingURL=sessions.js.map