Object.defineProperty(exports, "__esModule", { value: true });
exports.configureIPC = void 0;
const core_1 = require("@sentry/core");
const utils_1 = require("@sentry/utils");
const electron_1 = require("electron");
const util_1 = require("util");
const common_1 = require("../common");
const electron_normalize_1 = require("./electron-normalize");
function captureEventFromRenderer(options, event, attachments, contents) {
    var _a, _b, _c, _d;
    const process = contents ? ((_a = options === null || options === void 0 ? void 0 : options.getRendererName) === null || _a === void 0 ? void 0 : _a.call(options, contents)) || 'renderer' : 'renderer';
    // Ensure breadcrumbs are empty as they sent via scope updates
    event.breadcrumbs = event.breadcrumbs || [];
    // Remove the environment as it defaults to 'production' and overwrites the main process environment
    delete event.environment;
    // Remove the SDK info as we want the Electron SDK to be the one reporting the event
    (_b = event.sdk) === null || _b === void 0 ? true : delete _b.name;
    (_c = event.sdk) === null || _c === void 0 ? true : delete _c.version;
    (_d = event.sdk) === null || _d === void 0 ? true : delete _d.packages;
    (0, core_1.captureEvent)((0, common_1.mergeEvents)(event, { tags: { 'event.process': process } }), { attachments });
}
function handleEvent(options, jsonEvent, contents) {
    let event;
    try {
        event = JSON.parse(jsonEvent);
    }
    catch (_a) {
        utils_1.logger.warn('sentry-electron received an invalid event message');
        return;
    }
    captureEventFromRenderer(options, event, [], contents);
}
function eventFromEnvelope(envelope) {
    let event;
    const attachments = [];
    (0, utils_1.forEachEnvelopeItem)(envelope, (item, type) => {
        if (type === 'event' || type === 'transaction') {
            event = Array.isArray(item) ? item[1] : undefined;
        }
        else if (type === 'attachment') {
            const [headers, data] = item;
            attachments.push({
                filename: headers.filename,
                attachmentType: headers.attachment_type,
                contentType: headers.content_type,
                data,
            });
        }
    });
    return event ? [event, attachments] : undefined;
}
function handleEnvelope(options, env, contents) {
    var _a, _b;
    const envelope = (0, utils_1.parseEnvelope)(env, new util_1.TextEncoder(), new util_1.TextDecoder());
    const eventAndAttachments = eventFromEnvelope(envelope);
    if (eventAndAttachments) {
        const [event, attachments] = eventAndAttachments;
        captureEventFromRenderer(options, event, attachments, contents);
    }
    else {
        const normalizedEnvelope = (0, common_1.normalizeUrlsInReplayEnvelope)(envelope, electron_1.app.getAppPath());
        // Pass other types of envelope straight to the transport
        void ((_b = (_a = (0, core_1.getCurrentHub)().getClient()) === null || _a === void 0 ? void 0 : _a.getTransport()) === null || _b === void 0 ? void 0 : _b.send(normalizedEnvelope));
    }
}
/** Is object defined and has keys */
function hasKeys(obj) {
    return obj != undefined && Object.keys(obj).length > 0;
}
/**
 * Handle scope updates from renderer processes
 */
function handleScope(options, jsonScope) {
    let rendererScope;
    try {
        rendererScope = JSON.parse(jsonScope);
    }
    catch (_a) {
        utils_1.logger.warn('sentry-electron received an invalid scope message');
        return;
    }
    const sentScope = core_1.Scope.clone(rendererScope);
    /* eslint-disable @typescript-eslint/no-unsafe-member-access */
    (0, core_1.configureScope)((scope) => {
        if (hasKeys(sentScope._user)) {
            scope.setUser(sentScope._user);
        }
        if (hasKeys(sentScope._tags)) {
            scope.setTags(sentScope._tags);
        }
        if (hasKeys(sentScope._extra)) {
            scope.setExtras(sentScope._extra);
        }
        for (const attachment of sentScope._attachments || []) {
            scope.addAttachment(attachment);
        }
        const breadcrumb = sentScope._breadcrumbs.pop();
        if (breadcrumb) {
            scope.addBreadcrumb(breadcrumb, (options === null || options === void 0 ? void 0 : options.maxBreadcrumbs) || 100);
        }
    });
    /* eslint-enable @typescript-eslint/no-unsafe-member-access */
}
/** Enables Electron protocol handling */
function configureProtocol(options) {
    if (electron_1.app.isReady()) {
        throw new utils_1.SentryError("Sentry SDK should be initialized before the Electron app 'ready' event is fired");
    }
    electron_1.protocol.registerSchemesAsPrivileged([
        {
            scheme: common_1.PROTOCOL_SCHEME,
            privileges: { bypassCSP: true, corsEnabled: true, supportFetchAPI: true },
        },
    ]);
    electron_normalize_1.whenAppReady
        .then(() => {
        for (const sesh of options.getSessions()) {
            sesh.protocol.registerStringProtocol(common_1.PROTOCOL_SCHEME, (request, callback) => {
                var _a, _b;
                const data = (_b = (_a = request.uploadData) === null || _a === void 0 ? void 0 : _a[0]) === null || _b === void 0 ? void 0 : _b.bytes;
                if (request.url.startsWith(`${common_1.PROTOCOL_SCHEME}://${common_1.IPCChannel.EVENT}`) && data) {
                    handleEvent(options, data.toString());
                }
                else if (request.url.startsWith(`${common_1.PROTOCOL_SCHEME}://${common_1.IPCChannel.SCOPE}`) && data) {
                    handleScope(options, data.toString());
                }
                else if (request.url.startsWith(`${common_1.PROTOCOL_SCHEME}://${common_1.IPCChannel.ENVELOPE}`) && data) {
                    handleEnvelope(options, data);
                }
                callback('');
            });
        }
    })
        .catch((error) => utils_1.logger.error(error));
}
/**
 * Hooks IPC for communication with the renderer processes
 */
function configureClassic(options) {
    electron_1.ipcMain.on(common_1.IPCChannel.EVENT, ({ sender }, jsonEvent) => handleEvent(options, jsonEvent, sender));
    electron_1.ipcMain.on(common_1.IPCChannel.SCOPE, (_, jsonScope) => handleScope(options, jsonScope));
    electron_1.ipcMain.on(common_1.IPCChannel.ENVELOPE, ({ sender }, env) => handleEnvelope(options, env, sender));
}
/** Sets up communication channels with the renderer */
function configureIPC(options) {
    if (!(0, electron_normalize_1.supportsFullProtocol)() && options.ipcMode === common_1.IPCMode.Protocol) {
        throw new utils_1.SentryError('IPCMode.Protocol is only supported in Electron >= v5');
    }
    // eslint-disable-next-line no-bitwise
    if ((0, electron_normalize_1.supportsFullProtocol)() && (options.ipcMode & common_1.IPCMode.Protocol) > 0) {
        configureProtocol(options);
    }
    // eslint-disable-next-line no-bitwise
    if ((options.ipcMode & common_1.IPCMode.Classic) > 0) {
        configureClassic(options);
    }
}
exports.configureIPC = configureIPC;
//# sourceMappingURL=ipc.js.map