Object.defineProperty(exports, "__esModule", { value: true });
exports.getMinidumpLoader = exports.deleteMinidump = void 0;
const tslib_1 = require("tslib");
const utils_1 = require("@sentry/utils");
const path_1 = require("path");
const electron_normalize_1 = require("../../electron-normalize");
const fs_1 = require("../../fs");
/** Maximum number of days to keep a minidump before deleting it. */
const MAX_AGE = 30;
function filterAsync(array, predicate, thisArg) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const verdicts = yield Promise.all(array.map(predicate, thisArg));
        return array.filter((_, index) => verdicts[index]);
    });
}
/** Deletes a minidump */
function deleteMinidump(dump) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            yield (0, fs_1.unlinkAsync)(dump.path);
        }
        catch (e) {
            utils_1.logger.warn('Could not delete', dump.path);
        }
    });
}
exports.deleteMinidump = deleteMinidump;
function createMinidumpLoader(fetchMinidumpsImpl) {
    const knownPaths = [];
    return () => tslib_1.__awaiter(this, void 0, void 0, function* () {
        const minidumps = yield fetchMinidumpsImpl();
        utils_1.logger.log(`Found ${minidumps.length} minidumps`);
        const oldestMs = new Date().getTime() - MAX_AGE * 24 * 3600 * 1000;
        return filterAsync(minidumps, (dump) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            // Skip files that we have seen before
            if (knownPaths.indexOf(dump.path) >= 0) {
                return false;
            }
            // Lock this minidump until we have uploaded it or an error occurs and we
            // remove it from the file system.
            knownPaths.push(dump.path);
            // We do not want to upload minidumps that have been generated before a
            // certain threshold. Those old files can be deleted immediately.
            const stats = yield (0, fs_1.statAsync)(dump.path);
            if (stats.birthtimeMs < oldestMs) {
                yield deleteMinidump(dump);
                knownPaths.splice(knownPaths.indexOf(dump.path), 1);
                return false;
            }
            return true;
        }));
    });
}
/** Attempts to remove the metadata file so Crashpad doesn't output `failed to stat report` errors to the console */
function deleteCrashpadMetadataFile(crashesDirectory, waitMs = 100) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        if (waitMs > 2000) {
            return;
        }
        const metadataPath = (0, path_1.join)(crashesDirectory, 'metadata');
        try {
            yield (0, fs_1.unlinkAsync)(metadataPath);
            utils_1.logger.log('Deleted Crashpad metadata file', metadataPath);
        }
        catch (e) {
            // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
            if (e.code && e.code == 'EBUSY') {
                // Since Crashpad probably still has the metadata file open, we make a few attempts to delete it, backing
                // off and waiting longer each time.
                setTimeout(() => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield deleteCrashpadMetadataFile(crashesDirectory, waitMs * 2);
                }), waitMs);
            }
        }
    });
}
function readDirsAsync(paths) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const found = [];
        for (const path of paths) {
            try {
                const files = yield (0, fs_1.readDirAsync)(path);
                found.push(...files.map((file) => (0, path_1.join)(path, file)));
            }
            catch (_) {
                //
            }
        }
        return found;
    });
}
function crashpadMinidumpLoader() {
    const crashesDirectory = (0, electron_normalize_1.getCrashesDirectory)();
    const crashpadSubDirectory = process.platform === 'win32' ? 'reports' : 'completed';
    const dumpDirectories = [(0, path_1.join)(crashesDirectory, crashpadSubDirectory)];
    if (process.platform === 'darwin') {
        dumpDirectories.push((0, path_1.join)(crashesDirectory, 'pending'));
    }
    return createMinidumpLoader(() => tslib_1.__awaiter(this, void 0, void 0, function* () {
        yield deleteCrashpadMetadataFile(crashesDirectory).catch((error) => utils_1.logger.error(error));
        const files = yield readDirsAsync(dumpDirectories);
        return files
            .filter((file) => file.endsWith('.dmp'))
            .map((path) => {
            return {
                path,
                load: () => (0, fs_1.readFileAsync)(path),
            };
        });
    }));
}
/** Crudely parses the minidump from the Breakpad multipart file */
function minidumpFromBreakpadMultipart(file) {
    const binaryStart = file.lastIndexOf('Content-Type: application/octet-stream');
    if (binaryStart > 0) {
        const dumpStart = file.indexOf('MDMP', binaryStart);
        const dumpEnd = file.lastIndexOf('----------------------------');
        if (dumpStart > 0 && dumpEnd > 0 && dumpEnd > dumpStart) {
            return file.subarray(dumpStart, dumpEnd);
        }
    }
    return undefined;
}
function removeBreakpadMetadata(crashesDirectory, paths) {
    // Remove all metadata files and forget about them.
    void Promise.all(paths
        .filter((file) => file.endsWith('.txt') && !file.endsWith('log.txt'))
        .map((file) => tslib_1.__awaiter(this, void 0, void 0, function* () {
        const path = (0, path_1.join)(crashesDirectory, file);
        try {
            yield (0, fs_1.unlinkAsync)(path);
        }
        catch (e) {
            utils_1.logger.warn('Could not delete', path);
        }
    })));
}
function breakpadMinidumpLoader() {
    const crashesDirectory = (0, electron_normalize_1.getCrashesDirectory)();
    return createMinidumpLoader(() => tslib_1.__awaiter(this, void 0, void 0, function* () {
        // Breakpad stores all minidump files along with a metadata file directly in
        // the crashes directory.
        const files = yield (0, fs_1.readDirAsync)(crashesDirectory);
        removeBreakpadMetadata(crashesDirectory, files);
        return files
            .filter((file) => file.endsWith('.dmp'))
            .map((file) => {
            const path = (0, path_1.join)(crashesDirectory, file);
            return {
                path,
                load: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const file = yield (0, fs_1.readFileAsync)(path);
                    return minidumpFromBreakpadMultipart(file) || file;
                }),
            };
        })
            .filter((m) => !!m);
    }));
}
/**
 * Gets the minidump loader
 */
function getMinidumpLoader() {
    return (0, electron_normalize_1.usesCrashpad)() ? crashpadMinidumpLoader() : breakpadMinidumpLoader();
}
exports.getMinidumpLoader = getMinidumpLoader;
//# sourceMappingURL=minidump-loader.js.map