Object.defineProperty(exports, "__esModule", { value: true });
exports.MainProcessSession = void 0;
const tslib_1 = require("tslib");
const utils_1 = require("@sentry/utils");
const electron_1 = require("electron");
const sessions_1 = require("../sessions");
/** Tracks sessions as the main process lifetime. */
class MainProcessSession {
    constructor() {
        /** @inheritDoc */
        this.name = MainProcessSession.id;
        /** Handles the exit */
        this._exitHandler = (event) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (event.defaultPrevented) {
                return;
            }
            utils_1.logger.log('[MainProcessSession] Exit Handler');
            // Stop the exit so we have time to send the session
            event.preventDefault();
            try {
                // End the session
                yield (0, sessions_1.endSession)();
            }
            catch (e) {
                // Ignore and log any errors which would prevent app exit
                utils_1.logger.warn('[MainProcessSession] Error ending session:', e);
            }
            electron_1.app.exit();
        });
    }
    /** @inheritDoc */
    setupOnce() {
        void (0, sessions_1.startSession)();
        // We track sessions via the 'will-quit' event which is the last event emitted before close.
        //
        // We need to be the last 'will-quit' listener so as not to interfere with any user defined listeners which may
        // call `event.preventDefault()`.
        this._ensureExitHandlerLast();
        // 'before-quit' is always called before 'will-quit' so we listen there and ensure our 'will-quit' handler is still
        // the last listener
        electron_1.app.on('before-quit', () => {
            this._ensureExitHandlerLast();
        });
    }
    /**
     * Hooks 'will-quit' and ensures the handler is always last
     */
    _ensureExitHandlerLast() {
        electron_1.app.removeListener('will-quit', this._exitHandler);
        electron_1.app.on('will-quit', this._exitHandler);
    }
}
exports.MainProcessSession = MainProcessSession;
/** @inheritDoc */
MainProcessSession.id = 'MainProcessSession';
//# sourceMappingURL=main-process-session.js.map