Object.defineProperty(exports, "__esModule", { value: true });
exports.ElectronBreadcrumbs = exports.normalizeOptions = void 0;
const core_1 = require("@sentry/core");
const electron_1 = require("electron");
const electron_normalize_1 = require("../electron-normalize");
const renderers_1 = require("../renderers");
const DEFAULT_OPTIONS = {
    // We exclude events starting with remote as they can be quite verbose
    app: (name) => !name.startsWith('remote-'),
    autoUpdater: () => true,
    webContents: (name) => ['dom-ready', 'context-menu', 'load-url', 'destroyed'].includes(name),
    browserWindow: (name) => [
        'closed',
        'close',
        'unresponsive',
        'responsive',
        'show',
        'blur',
        'focus',
        'hide',
        'maximize',
        'minimize',
        'restore',
        'enter-full-screen',
        'leave-full-screen',
    ].includes(name),
    screen: () => true,
    powerMonitor: () => true,
    captureWindowTitles: false,
};
/** Converts all user supplied options to function | false */
function normalizeOptions(options) {
    return Object.keys(options).reduce((obj, k) => {
        if (k === 'captureWindowTitles') {
            obj[k] = !!options[k];
        }
        else {
            const val = options[k];
            if (Array.isArray(val)) {
                obj[k] = (name) => val.includes(name);
            }
            else if (typeof val === 'function' || val === false) {
                obj[k] = val;
            }
        }
        return obj;
    }, {});
}
exports.normalizeOptions = normalizeOptions;
/** Adds breadcrumbs for Electron events. */
class ElectronBreadcrumbs {
    /**
     * @param _options Integration options
     */
    constructor(options = {}) {
        /** @inheritDoc */
        this.name = ElectronBreadcrumbs.id;
        this._options = Object.assign(Object.assign({}, DEFAULT_OPTIONS), normalizeOptions(options));
    }
    /** @inheritDoc */
    setupOnce() {
        var _a;
        const initOptions = (_a = (0, core_1.getCurrentHub)().getClient()) === null || _a === void 0 ? void 0 : _a.getOptions();
        (0, renderers_1.trackRendererProperties)();
        void electron_normalize_1.whenAppReady.then(() => {
            // We can't access these until app 'ready'
            if (this._options.screen) {
                this._patchEventEmitter(electron_1.screen, 'screen', this._options.screen);
            }
            if (this._options.powerMonitor) {
                this._patchEventEmitter(electron_1.powerMonitor, 'powerMonitor', this._options.powerMonitor);
            }
        });
        if (this._options.app) {
            this._patchEventEmitter(electron_1.app, 'app', this._options.app);
        }
        if (this._options.autoUpdater) {
            this._patchEventEmitter(electron_1.autoUpdater, 'autoUpdater', this._options.autoUpdater);
        }
        if (this._options.browserWindow) {
            (0, electron_normalize_1.onBrowserWindowCreated)((window) => {
                var _a;
                const id = window.webContents.id;
                const windowName = ((_a = initOptions === null || initOptions === void 0 ? void 0 : initOptions.getRendererName) === null || _a === void 0 ? void 0 : _a.call(initOptions, window.webContents)) || 'window';
                this._patchEventEmitter(window, windowName, this._options.browserWindow, id);
            });
        }
        if (this._options.webContents) {
            (0, electron_normalize_1.onWebContentsCreated)((contents) => {
                var _a;
                const id = contents.id;
                const webContentsName = ((_a = initOptions === null || initOptions === void 0 ? void 0 : initOptions.getRendererName) === null || _a === void 0 ? void 0 : _a.call(initOptions, contents)) || 'renderer';
                this._patchEventEmitter(contents, webContentsName, this._options.webContents, id);
            });
        }
    }
    /**
     * Monkey patches the EventEmitter to capture breadcrumbs for the specified events. 🙈
     */
    _patchEventEmitter(emitter, category, shouldCapture, id) {
        const emit = emitter.emit.bind(emitter);
        emitter.emit = (event, ...args) => {
            var _a, _b;
            if (shouldCapture && shouldCapture(event)) {
                const breadcrumb = {
                    category: 'electron',
                    message: `${category}.${event}`,
                    timestamp: new Date().getTime() / 1000,
                    type: 'ui',
                };
                if (id) {
                    breadcrumb.data = Object.assign({}, (0, renderers_1.getRendererProperties)(id));
                    if (!this._options.captureWindowTitles && ((_a = breadcrumb.data) === null || _a === void 0 ? void 0 : _a.title)) {
                        (_b = breadcrumb.data) === null || _b === void 0 ? true : delete _b.title;
                    }
                }
                (0, core_1.addBreadcrumb)(breadcrumb);
            }
            return emit(event, ...args);
        };
    }
}
exports.ElectronBreadcrumbs = ElectronBreadcrumbs;
/** @inheritDoc */
ElectronBreadcrumbs.id = 'ElectronBreadcrumbs';
//# sourceMappingURL=electron-breadcrumbs.js.map