import { __awaiter } from "tslib";
import { getCurrentHub, makeSession, updateSession } from '@sentry/core';
import { flush } from '@sentry/node';
import { logger } from '@sentry/utils';
import { sentryCachePath } from './fs';
import { Store } from './store';
const PERSIST_INTERVAL_MS = 60000;
/** Stores the app session in case of termination due to main process crash or app killed */
const sessionStore = new Store(sentryCachePath, 'session', undefined);
/** Previous session that did not exit cleanly */
let previousSession = sessionStore.get();
let persistTimer;
/** Starts a session */
export function startSession() {
    return __awaiter(this, void 0, void 0, function* () {
        const hub = getCurrentHub();
        yield sessionStore.set(hub.startSession());
        // Every PERSIST_INTERVAL, write the session to disk
        persistTimer = setInterval(() => __awaiter(this, void 0, void 0, function* () {
            var _a;
            const currentSession = (_a = hub.getScope()) === null || _a === void 0 ? void 0 : _a.getSession();
            // Only bother saving if it hasn't already ended
            if (currentSession && currentSession.status === 'ok') {
                yield sessionStore.set(currentSession);
            }
        }), PERSIST_INTERVAL_MS);
    });
}
/** Cleanly ends a session */
export function endSession() {
    var _a;
    return __awaiter(this, void 0, void 0, function* () {
        // Once the session had ended there is no point persisting it
        if (persistTimer) {
            clearInterval(persistTimer);
        }
        const hub = getCurrentHub();
        const session = (_a = hub.getScope()) === null || _a === void 0 ? void 0 : _a.getSession();
        if (session) {
            if (session.status === 'ok') {
                logger.log('Ending session');
                hub.endSession();
            }
            else {
                logger.log('Session was already ended');
            }
        }
        else {
            logger.log('No session');
        }
        yield sessionStore.clear();
        yield flush();
    });
}
/** Determines if a Date is likely to have occurred in the previous uncompleted session */
export function unreportedDuringLastSession(crashDate) {
    return __awaiter(this, void 0, void 0, function* () {
        if (!crashDate) {
            return false;
        }
        const previousSessionModified = yield sessionStore.getModifiedDate();
        // There is no previous session
        if (previousSessionModified == undefined) {
            return false;
        }
        const previousSessionModifiedTime = previousSessionModified.getTime();
        const crashTime = crashDate.getTime();
        // Session could have run until modified time + persist interval
        const prevSessionEnd = previousSessionModifiedTime + PERSIST_INTERVAL_MS;
        // Event cannot have occurred before last persist time, We add a 2 second overlap to be sure
        const lastPersist = previousSessionModifiedTime - 2000;
        // If the crash occurred between the last persist and estimated end of session
        return crashTime > lastPersist && crashTime < prevSessionEnd;
    });
}
/** Checks if the previous session needs sending as crashed or abnormal  */
export function checkPreviousSession(crashed) {
    var _a, _b;
    return __awaiter(this, void 0, void 0, function* () {
        const client = getCurrentHub().getClient();
        const previous = yield previousSession;
        if (previous && client) {
            // Ignore if the previous session is already ended
            if (previous.status !== 'ok') {
                previousSession = undefined;
                return;
            }
            const status = crashed ? 'crashed' : 'abnormal';
            logger.log(`Found previous ${status} session`);
            const sesh = makeSession(previous);
            updateSession(sesh, {
                status,
                errors: (sesh.errors || 0) + 1,
                release: (_a = previous.attrs) === null || _a === void 0 ? void 0 : _a.release,
                environment: (_b = previous.attrs) === null || _b === void 0 ? void 0 : _b.environment,
            });
            yield client.sendSession(sesh);
            previousSession = undefined;
        }
    });
}
/** Sets the current session as crashed */
export function sessionCrashed() {
    var _a;
    // stop persisting session
    if (persistTimer) {
        clearInterval(persistTimer);
    }
    logger.log('Session Crashed');
    const hub = getCurrentHub();
    const session = (_a = hub.getScope()) === null || _a === void 0 ? void 0 : _a.getSession();
    if (!session) {
        logger.log('No session to update');
        return;
    }
    if (session.status === 'ok') {
        logger.log('Setting session as crashed');
        updateSession(session, { status: 'crashed', errors: (session.errors += 1) });
    }
    else {
        logger.log('Session already ended');
    }
    hub.captureSession();
}
//# sourceMappingURL=sessions.js.map