import { __awaiter } from "tslib";
import { logger } from '@sentry/utils';
import { join } from 'path';
import { getCrashesDirectory, usesCrashpad } from '../../electron-normalize';
import { readDirAsync, readFileAsync, statAsync, unlinkAsync } from '../../fs';
/** Maximum number of days to keep a minidump before deleting it. */
const MAX_AGE = 30;
function filterAsync(array, predicate, thisArg) {
    return __awaiter(this, void 0, void 0, function* () {
        const verdicts = yield Promise.all(array.map(predicate, thisArg));
        return array.filter((_, index) => verdicts[index]);
    });
}
/** Deletes a minidump */
export function deleteMinidump(dump) {
    return __awaiter(this, void 0, void 0, function* () {
        try {
            yield unlinkAsync(dump.path);
        }
        catch (e) {
            logger.warn('Could not delete', dump.path);
        }
    });
}
function createMinidumpLoader(fetchMinidumpsImpl) {
    const knownPaths = [];
    return () => __awaiter(this, void 0, void 0, function* () {
        const minidumps = yield fetchMinidumpsImpl();
        logger.log(`Found ${minidumps.length} minidumps`);
        const oldestMs = new Date().getTime() - MAX_AGE * 24 * 3600 * 1000;
        return filterAsync(minidumps, (dump) => __awaiter(this, void 0, void 0, function* () {
            // Skip files that we have seen before
            if (knownPaths.indexOf(dump.path) >= 0) {
                return false;
            }
            // Lock this minidump until we have uploaded it or an error occurs and we
            // remove it from the file system.
            knownPaths.push(dump.path);
            // We do not want to upload minidumps that have been generated before a
            // certain threshold. Those old files can be deleted immediately.
            const stats = yield statAsync(dump.path);
            if (stats.birthtimeMs < oldestMs) {
                yield deleteMinidump(dump);
                knownPaths.splice(knownPaths.indexOf(dump.path), 1);
                return false;
            }
            return true;
        }));
    });
}
/** Attempts to remove the metadata file so Crashpad doesn't output `failed to stat report` errors to the console */
function deleteCrashpadMetadataFile(crashesDirectory, waitMs = 100) {
    return __awaiter(this, void 0, void 0, function* () {
        if (waitMs > 2000) {
            return;
        }
        const metadataPath = join(crashesDirectory, 'metadata');
        try {
            yield unlinkAsync(metadataPath);
            logger.log('Deleted Crashpad metadata file', metadataPath);
        }
        catch (e) {
            // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
            if (e.code && e.code == 'EBUSY') {
                // Since Crashpad probably still has the metadata file open, we make a few attempts to delete it, backing
                // off and waiting longer each time.
                setTimeout(() => __awaiter(this, void 0, void 0, function* () {
                    yield deleteCrashpadMetadataFile(crashesDirectory, waitMs * 2);
                }), waitMs);
            }
        }
    });
}
function readDirsAsync(paths) {
    return __awaiter(this, void 0, void 0, function* () {
        const found = [];
        for (const path of paths) {
            try {
                const files = yield readDirAsync(path);
                found.push(...files.map((file) => join(path, file)));
            }
            catch (_) {
                //
            }
        }
        return found;
    });
}
function crashpadMinidumpLoader() {
    const crashesDirectory = getCrashesDirectory();
    const crashpadSubDirectory = process.platform === 'win32' ? 'reports' : 'completed';
    const dumpDirectories = [join(crashesDirectory, crashpadSubDirectory)];
    if (process.platform === 'darwin') {
        dumpDirectories.push(join(crashesDirectory, 'pending'));
    }
    return createMinidumpLoader(() => __awaiter(this, void 0, void 0, function* () {
        yield deleteCrashpadMetadataFile(crashesDirectory).catch((error) => logger.error(error));
        const files = yield readDirsAsync(dumpDirectories);
        return files
            .filter((file) => file.endsWith('.dmp'))
            .map((path) => {
            return {
                path,
                load: () => readFileAsync(path),
            };
        });
    }));
}
/** Crudely parses the minidump from the Breakpad multipart file */
function minidumpFromBreakpadMultipart(file) {
    const binaryStart = file.lastIndexOf('Content-Type: application/octet-stream');
    if (binaryStart > 0) {
        const dumpStart = file.indexOf('MDMP', binaryStart);
        const dumpEnd = file.lastIndexOf('----------------------------');
        if (dumpStart > 0 && dumpEnd > 0 && dumpEnd > dumpStart) {
            return file.subarray(dumpStart, dumpEnd);
        }
    }
    return undefined;
}
function removeBreakpadMetadata(crashesDirectory, paths) {
    // Remove all metadata files and forget about them.
    void Promise.all(paths
        .filter((file) => file.endsWith('.txt') && !file.endsWith('log.txt'))
        .map((file) => __awaiter(this, void 0, void 0, function* () {
        const path = join(crashesDirectory, file);
        try {
            yield unlinkAsync(path);
        }
        catch (e) {
            logger.warn('Could not delete', path);
        }
    })));
}
function breakpadMinidumpLoader() {
    const crashesDirectory = getCrashesDirectory();
    return createMinidumpLoader(() => __awaiter(this, void 0, void 0, function* () {
        // Breakpad stores all minidump files along with a metadata file directly in
        // the crashes directory.
        const files = yield readDirAsync(crashesDirectory);
        removeBreakpadMetadata(crashesDirectory, files);
        return files
            .filter((file) => file.endsWith('.dmp'))
            .map((file) => {
            const path = join(crashesDirectory, file);
            return {
                path,
                load: () => __awaiter(this, void 0, void 0, function* () {
                    const file = yield readFileAsync(path);
                    return minidumpFromBreakpadMultipart(file) || file;
                }),
            };
        })
            .filter((m) => !!m);
    }));
}
/**
 * Gets the minidump loader
 */
export function getMinidumpLoader() {
    return usesCrashpad() ? crashpadMinidumpLoader() : breakpadMinidumpLoader();
}
//# sourceMappingURL=minidump-loader.js.map