import { __awaiter } from "tslib";
import { captureEvent, getCurrentHub, Scope } from '@sentry/core';
import { basename, logger, SentryError } from '@sentry/utils';
import { app, crashReporter } from 'electron';
import { mergeEvents } from '../../../common';
import { getEventDefaults } from '../../context';
import { EXIT_REASONS, onChildProcessGone, onRendererProcessGone } from '../../electron-normalize';
import { sentryCachePath } from '../../fs';
import { getRendererProperties, trackRendererProperties } from '../../renderers';
import { checkPreviousSession, sessionCrashed } from '../../sessions';
import { BufferedWriteStore } from '../../store';
import { deleteMinidump, getMinidumpLoader } from './minidump-loader';
/** Sends minidumps via the Sentry uploader */
export class SentryMinidump {
    constructor() {
        /** @inheritDoc */
        this.name = SentryMinidump.id;
    }
    /** @inheritDoc */
    setupOnce() {
        // Mac AppStore builds cannot run the crash reporter due to the sandboxing
        // requirements. In this case, we prevent enabling native crashes entirely.
        // https://electronjs.org/docs/tutorial/mac-app-store-submission-guide#limitations-of-mas-build
        if (process.mas) {
            return;
        }
        this._startCrashReporter();
        this._scopeStore = new BufferedWriteStore(sentryCachePath, 'scope_v2', new Scope());
        // We need to store the scope in a variable here so it can be attached to minidumps
        this._scopeLastRun = this._scopeStore.get();
        this._setupScopeListener();
        const client = getCurrentHub().getClient();
        const options = client === null || client === void 0 ? void 0 : client.getOptions();
        if (!(options === null || options === void 0 ? void 0 : options.dsn)) {
            throw new SentryError('Attempted to enable Electron native crash reporter but no DSN was supplied');
        }
        trackRendererProperties();
        this._minidumpLoader = getMinidumpLoader();
        onRendererProcessGone(EXIT_REASONS, (contents, details) => this._sendRendererCrash(options, contents, details));
        onChildProcessGone(EXIT_REASONS, (details) => this._sendChildProcessCrash(options, details));
        // Start to submit recent minidump crashes. This will load breadcrumbs and
        // context information that was cached on disk prior to the crash.
        this._sendNativeCrashes({
            level: 'fatal',
            platform: 'native',
            tags: {
                'event.environment': 'native',
                'event.process': 'browser',
                event_type: 'native',
            },
        })
            .then((minidumpsFound) => 
        // Check for previous uncompleted session. If a previous session exists
        // and no minidumps were found, its likely an abnormal exit
        checkPreviousSession(minidumpsFound))
            .catch((error) => logger.error(error));
    }
    /** Starts the native crash reporter */
    _startCrashReporter() {
        logger.log('Starting Electron crashReporter');
        crashReporter.start({
            companyName: '',
            ignoreSystemCrashHandler: true,
            productName: app.name || app.getName(),
            // Empty string doesn't work for Linux Crashpad and no submitURL doesn't work for older versions of Electron
            submitURL: 'https://f.a.k/e',
            uploadToServer: false,
            compress: true,
        });
    }
    /**
     * Helper function for sending renderer crashes
     */
    _sendRendererCrash(options, contents, details) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            const { getRendererName, release, environment } = options;
            const crashedProcess = (getRendererName === null || getRendererName === void 0 ? void 0 : getRendererName(contents)) || 'renderer';
            logger.log(`'${crashedProcess}' process '${details.reason}'`);
            const event = mergeEvents(yield getEventDefaults(release, environment), {
                contexts: {
                    electron: {
                        crashed_url: ((_a = getRendererProperties(contents.id)) === null || _a === void 0 ? void 0 : _a.url) || 'unknown',
                        details,
                    },
                },
                level: 'fatal',
                // The default is javascript
                platform: 'native',
                tags: {
                    'event.environment': 'native',
                    'event.process': crashedProcess,
                    'exit.reason': details.reason,
                    event_type: 'native',
                },
            });
            const found = yield this._sendNativeCrashes(event);
            if (found) {
                sessionCrashed();
            }
        });
    }
    /**
     * Helper function for sending child process crashes
     */
    _sendChildProcessCrash(options, details) {
        return __awaiter(this, void 0, void 0, function* () {
            logger.log(`${details.type} process has ${details.reason}`);
            const { release, environment } = options;
            const event = mergeEvents(yield getEventDefaults(release, environment), {
                contexts: {
                    electron: { details },
                },
                level: 'fatal',
                // The default is javascript
                platform: 'native',
                tags: {
                    'event.environment': 'native',
                    'event.process': details.type,
                    'exit.reason': details.reason,
                    event_type: 'native',
                },
            });
            const found = yield this._sendNativeCrashes(event);
            if (found) {
                sessionCrashed();
            }
        });
    }
    /**
     * Adds a scope listener to persist changes to disk.
     */
    _setupScopeListener() {
        const hubScope = getCurrentHub().getScope();
        if (hubScope) {
            hubScope.addScopeListener((updatedScope) => {
                const scope = Scope.clone(updatedScope);
                // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
                scope._eventProcessors = [];
                // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
                scope._scopeListeners = [];
                // Since the initial scope read is async, we need to ensure that any writes do not beat that
                // https://github.com/getsentry/sentry-electron/issues/585
                setImmediate(() => {
                    var _a;
                    void ((_a = this._scopeStore) === null || _a === void 0 ? void 0 : _a.set(scope));
                });
            });
        }
    }
    /**
     * Loads new native crashes from disk and sends them to Sentry.
     *
     * Returns true if one or more minidumps were found
     */
    _sendNativeCrashes(event) {
        return __awaiter(this, void 0, void 0, function* () {
            // Whenever we are called, assume that the crashes we are going to load down
            // below have occurred recently. This means, we can use the same event data
            // for all minidumps that we load now. There are two conditions:
            //
            //  1. The application crashed and we are just starting up. The stored
            //     breadcrumbs and context reflect the state during the application
            //     crash.
            //
            //  2. A renderer process crashed recently and we have just been notified
            //     about it. Just use the breadcrumbs and context information we have
            //     right now and hope that the delay was not too long.
            if (this._minidumpLoader === undefined) {
                throw new SentryError('Invariant violation: Native crashes not enabled');
            }
            try {
                const minidumps = yield this._minidumpLoader();
                if (minidumps.length > 0) {
                    const hub = getCurrentHub();
                    const client = hub.getClient();
                    if (!client) {
                        return true;
                    }
                    const enabled = client.getOptions().enabled;
                    // If the SDK is not enabled, we delete the minidump files so they
                    // dont accumulate and/or get sent later
                    if (enabled === false) {
                        minidumps.forEach(deleteMinidump);
                        return false;
                    }
                    const storedScope = Scope.clone(yield this._scopeLastRun);
                    let newEvent = yield storedScope.applyToEvent(event);
                    const hubScope = hub.getScope();
                    newEvent = hubScope ? yield hubScope.applyToEvent(event) : event;
                    if (!newEvent) {
                        return false;
                    }
                    for (const minidump of minidumps) {
                        const data = yield minidump.load();
                        if (data) {
                            captureEvent(newEvent, {
                                attachments: [
                                    {
                                        attachmentType: 'event.minidump',
                                        filename: basename(minidump.path),
                                        data,
                                    },
                                ],
                            });
                        }
                        void deleteMinidump(minidump);
                    }
                    // Unset to recover memory
                    this._scopeLastRun = undefined;
                    return true;
                }
            }
            catch (_oO) {
                logger.error('Error while sending native crash.');
            }
            return false;
        });
    }
}
/** @inheritDoc */
SentryMinidump.id = 'SentryMinidump';
//# sourceMappingURL=index.js.map