import { __awaiter } from "tslib";
import { logger } from '@sentry/utils';
import { app } from 'electron';
import { endSession, startSession } from '../sessions';
/** Tracks sessions as the main process lifetime. */
export class MainProcessSession {
    constructor() {
        /** @inheritDoc */
        this.name = MainProcessSession.id;
        /** Handles the exit */
        this._exitHandler = (event) => __awaiter(this, void 0, void 0, function* () {
            if (event.defaultPrevented) {
                return;
            }
            logger.log('[MainProcessSession] Exit Handler');
            // Stop the exit so we have time to send the session
            event.preventDefault();
            try {
                // End the session
                yield endSession();
            }
            catch (e) {
                // Ignore and log any errors which would prevent app exit
                logger.warn('[MainProcessSession] Error ending session:', e);
            }
            app.exit();
        });
    }
    /** @inheritDoc */
    setupOnce() {
        void startSession();
        // We track sessions via the 'will-quit' event which is the last event emitted before close.
        //
        // We need to be the last 'will-quit' listener so as not to interfere with any user defined listeners which may
        // call `event.preventDefault()`.
        this._ensureExitHandlerLast();
        // 'before-quit' is always called before 'will-quit' so we listen there and ensure our 'will-quit' handler is still
        // the last listener
        app.on('before-quit', () => {
            this._ensureExitHandlerLast();
        });
    }
    /**
     * Hooks 'will-quit' and ensures the handler is always last
     */
    _ensureExitHandlerLast() {
        app.removeListener('will-quit', this._exitHandler);
        app.on('will-quit', this._exitHandler);
    }
}
/** @inheritDoc */
MainProcessSession.id = 'MainProcessSession';
//# sourceMappingURL=main-process-session.js.map