import { __awaiter } from "tslib";
import { getCurrentHub, Scope } from '@sentry/core';
import { logger, makeDsn, SentryError } from '@sentry/utils';
import { app, crashReporter } from 'electron';
import { mergeEvents, normalizeEvent } from '../../common';
import { getEventDefaults } from '../context';
import { CRASH_REASONS, onRendererProcessGone, rendererRequiresCrashReporterStart, usesCrashpad, } from '../electron-normalize';
import { checkPreviousSession, sessionCrashed, unreportedDuringLastSession } from '../sessions';
/** Is object defined and has keys */
function hasKeys(obj) {
    return obj != undefined && Object.keys(obj).length > 0;
}
/** Gets a Scope object with user, tags and extra */
function getScope(options) {
    const scope = getCurrentHub().getScope();
    if (!scope) {
        return {};
    }
    return Object.assign(Object.assign(Object.assign({ release: options.release, environment: options.environment }, (hasKeys(scope._user) && { user: scope._user })), (hasKeys(scope._tags) && { tags: scope._tags })), (hasKeys(scope._extra) && { extra: scope._extra }));
}
/**
 * Returns the minidump endpoint in Sentry
 * @param dsn Dsn
 */
export function minidumpUrlFromDsn(dsn) {
    const { host, path, projectId, port, protocol, publicKey } = makeDsn(dsn);
    return `${protocol}://${host}${port !== '' ? `:${port}` : ''}${path !== '' ? `/${path}` : ''}/api/${projectId}/minidump/?sentry_key=${publicKey}`;
}
/** Sends minidumps via the Electron built-in uploader. */
export class ElectronMinidump {
    constructor() {
        /** @inheritDoc */
        this.name = ElectronMinidump.id;
        /** Counter used to ensure no race condition when updating extra params */
        this._updateEpoch = 0;
    }
    /** @inheritDoc */
    setupOnce() {
        var _a, _b;
        // Mac AppStore builds cannot run the crash reporter due to the sandboxing
        // requirements. In this case, we prevent enabling native crashes entirely.
        // https://electronjs.org/docs/tutorial/mac-app-store-submission-guide#limitations-of-mas-build
        if (process.mas) {
            return;
        }
        if (rendererRequiresCrashReporterStart()) {
            throw new SentryError(`The '${this.name}' integration is only supported with Electron >= v9`);
        }
        const options = (_a = getCurrentHub().getClient()) === null || _a === void 0 ? void 0 : _a.getOptions();
        if (!(options === null || options === void 0 ? void 0 : options.dsn)) {
            throw new SentryError('Attempted to enable Electron native crash reporter but no DSN was supplied');
        }
        this._customRelease = options.release;
        this._startCrashReporter(options);
        // If a renderer process crashes, mark any existing session as crashed
        onRendererProcessGone(CRASH_REASONS, (_, __) => {
            sessionCrashed();
        });
        // If we're using the Crashpad minidump uploader, we set extra parameters whenever the scope updates
        if (usesCrashpad()) {
            this._setupScopeListener();
        }
        // Check if last crash report was likely to have been unreported in the last session
        void unreportedDuringLastSession((_b = crashReporter.getLastCrashReport()) === null || _b === void 0 ? void 0 : _b.date).then((crashed) => {
            // Check if a previous session was not closed
            checkPreviousSession(crashed).catch((error) => logger.error(error));
        });
    }
    /**
     * Starts the native crash reporter
     */
    _startCrashReporter(options) {
        // We don't add globalExtra when Breakpad is in use because it doesn't support JSON like strings:
        // https://github.com/electron/electron/issues/29711
        const globalExtra = usesCrashpad() ? { sentry___initialScope: JSON.stringify(getScope(options)) } : undefined;
        logger.log('Starting Electron crashReporter');
        crashReporter.start({
            companyName: '',
            ignoreSystemCrashHandler: true,
            productName: app.name || app.getName(),
            submitURL: minidumpUrlFromDsn(options.dsn || ''),
            uploadToServer: true,
            compress: true,
            globalExtra,
        });
    }
    /**
     * Adds a scope listener to persist changes to disk.
     */
    _setupScopeListener() {
        const hubScope = getCurrentHub().getScope();
        if (hubScope) {
            hubScope.addScopeListener((updatedScope) => {
                const scope = Scope.clone(updatedScope);
                /* eslint-disable @typescript-eslint/no-unsafe-member-access */
                scope._eventProcessors = [];
                scope._scopeListeners = [];
                /* eslint-enable @typescript-eslint/no-unsafe-member-access */
                this._updateExtraParams(scope);
            });
        }
    }
    /** Updates Electron uploader extra params */
    _updateExtraParams(scope) {
        this._updateEpoch += 1;
        const currentEpoch = this._updateEpoch;
        this._getNativeUploaderEvent(scope)
            .then((event) => {
            if (currentEpoch !== this._updateEpoch)
                return;
            // Update the extra parameters in the main process
            const mainParams = this._getNativeUploaderExtraParams(event);
            for (const key of Object.keys(mainParams)) {
                crashReporter.addExtraParameter(key, mainParams[key]);
            }
        })
            .catch((error) => logger.error(error));
    }
    /** Builds up an event to send with the native Electron uploader */
    _getNativeUploaderEvent(scope) {
        return __awaiter(this, void 0, void 0, function* () {
            const event = mergeEvents(yield getEventDefaults(this._customRelease), {
                level: 'fatal',
                platform: 'native',
                tags: { 'event.environment': 'native', event_type: 'native' },
            });
            // Apply the scope to the event
            yield scope.applyToEvent(event);
            delete event.sdkProcessingMetadata;
            // Normalise paths
            return normalizeEvent(event, app.getAppPath());
        });
    }
    /** Chunks up event JSON into 1 or more parameters for use with the native Electron uploader
     *
     * Returns chunks with keys and values:
     * {
     *    sentry__1: '{ json...',
     *    sentry__2: 'more json...',
     *    sentry__x: 'end json }',
     * }
     */
    _getNativeUploaderExtraParams(event) {
        const maxBytes = 20300;
        /** Max chunk sizes are in bytes so we can't chunk by characters or UTF8 could bite us.
         *
         * We attempt to split by space (32) and double quote characters (34) as there are plenty in JSON
         * and they are guaranteed to not be the first byte of a multi-byte UTF8 character.
         */
        let buf = Buffer.from(JSON.stringify(event));
        const chunks = [];
        while (buf.length) {
            // Find last '"'
            let i = buf.lastIndexOf(34, maxBytes + 1);
            // Or find last ' '
            if (i < 0)
                i = buf.lastIndexOf(32, maxBytes + 1);
            // Or find first '"'
            if (i < 0)
                i = buf.indexOf(34, maxBytes);
            // Or find first ' '
            if (i < 0)
                i = buf.indexOf(32, maxBytes);
            // We couldn't find any space or quote chars so split at maxBytes and hope for the best 🤷‍♂️
            if (i < 0)
                i = maxBytes;
            chunks.push(buf.subarray(0, i + 1).toString());
            buf = buf.subarray(i + 1);
        }
        return chunks.reduce((acc, cur, i) => {
            acc[`sentry__${i + 1}`] = cur;
            return acc;
        }, {});
    }
}
/** @inheritDoc */
ElectronMinidump.id = 'ElectronMinidump';
//# sourceMappingURL=electron-minidump.js.map