import { addBreadcrumb, captureMessage, getCurrentHub } from '@sentry/core';
import { EXIT_REASONS, onChildProcessGone, onRendererProcessGone } from '../electron-normalize';
const DEFAULT_OPTIONS = {
    breadcrumbs: EXIT_REASONS,
    events: ['abnormal-exit', 'launch-failed', 'integrity-failure'],
};
/** Gets message and severity */
function getMessageAndSeverity(reason, proc) {
    const message = `'${proc}' process exited with '${reason}'`;
    switch (reason) {
        case 'abnormal-exit':
        case 'killed':
            return { message, level: 'warning' };
        case 'crashed':
        case 'oom':
        case 'launch-failed':
        case 'integrity-failure':
            return { message, level: 'fatal' };
        default:
            return { message, level: 'debug' };
    }
}
/** Adds breadcrumbs for Electron events. */
export class ChildProcess {
    /**
     * @param _options Integration options
     */
    constructor(options = {}) {
        /** @inheritDoc */
        this.name = ChildProcess.id;
        const { breadcrumbs, events } = options;
        this._options = {
            breadcrumbs: Array.isArray(breadcrumbs) ? breadcrumbs : breadcrumbs == false ? [] : DEFAULT_OPTIONS.breadcrumbs,
            events: Array.isArray(events) ? events : events == false ? [] : DEFAULT_OPTIONS.events,
        };
    }
    /** @inheritDoc */
    setupOnce() {
        var _a;
        const { breadcrumbs, events } = this._options;
        const allReasons = Array.from(new Set([...breadcrumbs, ...events]));
        // only hook these events if we're after more than just the unresponsive event
        if (allReasons.length > 0) {
            const options = (_a = getCurrentHub().getClient()) === null || _a === void 0 ? void 0 : _a.getOptions();
            onChildProcessGone(allReasons, (details) => {
                const { reason } = details;
                // Capture message first
                if (events.includes(reason)) {
                    const { message, level } = getMessageAndSeverity(details.reason, details.type);
                    captureMessage(message, { level, tags: { 'event.process': details.type } });
                }
                // And then add breadcrumbs for subsequent events
                if (breadcrumbs.includes(reason)) {
                    addBreadcrumb(Object.assign(Object.assign({ type: 'process', category: 'child-process' }, getMessageAndSeverity(details.reason, details.type)), { data: details }));
                }
            });
            onRendererProcessGone(allReasons, (contents, details) => {
                var _a;
                const { reason } = details;
                const name = ((_a = options === null || options === void 0 ? void 0 : options.getRendererName) === null || _a === void 0 ? void 0 : _a.call(options, contents)) || 'renderer';
                // Capture message first
                if (events.includes(reason)) {
                    const { message, level } = getMessageAndSeverity(details.reason, name);
                    captureMessage(message, level);
                }
                // And then add breadcrumbs for subsequent events
                if (breadcrumbs.includes(reason)) {
                    addBreadcrumb(Object.assign(Object.assign({ type: 'process', category: 'child-process' }, getMessageAndSeverity(details.reason, name)), { data: details }));
                }
            });
        }
    }
}
/** @inheritDoc */
ChildProcess.id = 'ChildProcess';
//# sourceMappingURL=child-process.js.map