import { __awaiter } from "tslib";
import { app, screen as electronScreen } from 'electron';
import { cpus } from 'os';
import { mergeEvents } from '../../common';
import { whenAppReady } from '../electron-normalize';
const DEFAULT_OPTIONS = {
    cpu: true,
    screen: true,
    memory: true,
    language: true,
};
/** Adds Electron context to events and normalises paths. */
export class AdditionalContext {
    constructor(options = {}) {
        /** @inheritDoc */
        this.name = AdditionalContext.id;
        this._lazyDeviceContext = {};
        this._options = Object.assign(Object.assign({}, DEFAULT_OPTIONS), options);
    }
    /** @inheritDoc */
    setupOnce(addGlobalEventProcessor) {
        addGlobalEventProcessor((event) => __awaiter(this, void 0, void 0, function* () { return this._addAdditionalContext(event); }));
        // Some metrics are only available after app ready so we lazily load them
        void whenAppReady.then(() => {
            const { language, screen } = this._options;
            if (language) {
                this._lazyDeviceContext.language = app.getLocale();
            }
            if (screen) {
                this._setPrimaryDisplayInfo();
                electronScreen.on('display-metrics-changed', () => {
                    this._setPrimaryDisplayInfo();
                });
            }
        });
    }
    /** Adds additional context to event */
    _addAdditionalContext(event) {
        const device = this._lazyDeviceContext;
        const { memory, cpu } = this._options;
        if (memory) {
            const { total, free } = process.getSystemMemoryInfo();
            device.memory_size = total * 1024;
            device.free_memory = free * 1024;
        }
        if (cpu) {
            const cpuInfo = cpus();
            if (cpuInfo && cpuInfo.length) {
                const firstCpu = cpuInfo[0];
                device.processor_count = cpuInfo.length;
                device.cpu_description = firstCpu.model;
                device.processor_frequency = firstCpu.speed;
                if (app.runningUnderARM64Translation) {
                    device.machine_arch = 'arm64';
                }
            }
        }
        return mergeEvents(event, { contexts: { device } });
    }
    /** Sets the display info */
    _setPrimaryDisplayInfo() {
        const display = electronScreen.getPrimaryDisplay();
        const width = Math.floor(display.size.width * display.scaleFactor);
        const height = Math.floor(display.size.height * display.scaleFactor);
        this._lazyDeviceContext.screen_density = display.scaleFactor;
        this._lazyDeviceContext.screen_resolution = `${width}x${height}`;
    }
}
/** @inheritDoc */
AdditionalContext.id = 'AdditionalContext';
//# sourceMappingURL=additional-context.js.map