// based on https://github.com/facebook/create-react-app/blob/master/packages/react-dev-utils/openBrowser.js
import { execSync } from 'node:child_process';
import path from 'node:path';
import { fileURLToPath } from 'node:url';
import spawn from 'cross-spawn';
import open from 'open';
import picocolors from 'picocolors';
// https://github.com/sindresorhus/open#app
const OSX_CHROME = 'google chrome';
var Action;
(function (Action) {
    Action[Action["NONE"] = 0] = "NONE";
    Action[Action["BROWSER"] = 1] = "BROWSER";
    Action[Action["SCRIPT"] = 2] = "SCRIPT";
})(Action || (Action = {}));
function getBrowserEnv() {
    // Attempt to honor this environment variable.
    // It is specific to the operating system.
    // See https://github.com/sindresorhus/open#app for documentation.
    const value = process.env.BROWSER;
    const args = process.env.BROWSER_ARGS
        ? process.env.BROWSER_ARGS.split(' ')
        : [];
    let action;
    if (!value) {
        // Default.
        action = Action.BROWSER;
    }
    else if (value.toLowerCase().endsWith('.js')) {
        action = Action.SCRIPT;
    }
    else if (value.toLowerCase() === 'none') {
        action = Action.NONE;
    }
    else {
        action = Action.BROWSER;
    }
    return { action, value, args };
}
function executeNodeScript(scriptPath, url) {
    const extraArgs = process.argv.slice(2);
    const child = spawn(process.execPath, [scriptPath, ...extraArgs, url], {
        stdio: 'inherit',
    });
    child.on('close', code => {
        if (code !== 0) {
            console.log();
            console.log(picocolors.red('The script specified as BROWSER environment variable failed.'));
            console.log(`${picocolors.cyan(scriptPath)} exited with code ${code}`);
            console.log();
        }
    });
    return true;
}
function startBrowserProcess(browser, url, args) {
    // If we're on OS X, the user hasn't specifically
    // requested a different browser, we can try opening
    // Chrome with AppleScript. This lets us reuse an
    // existing tab when possible instead of creating a new one.
    const shouldTryOpenChromiumWithAppleScript = process.platform === 'darwin' &&
        (typeof browser !== 'string' || browser === OSX_CHROME);
    if (shouldTryOpenChromiumWithAppleScript) {
        // Will use the first open browser found from list
        const supportedChromiumBrowsers = [
            'Google Chrome Canary',
            'Google Chrome',
            'Microsoft Edge',
            'Brave Browser',
            'Vivaldi',
            'Chromium',
        ];
        const _dirname = typeof __dirname === 'undefined'
            ? path.dirname(fileURLToPath(import.meta.url))
            : __dirname;
        for (const chromiumBrowser of supportedChromiumBrowsers) {
            try {
                // Try our best to reuse existing tab
                // on OSX Chromium-based browser with AppleScript
                execSync('ps cax | grep "' + chromiumBrowser + '"');
                execSync('osascript ../openChrome.applescript "' +
                    encodeURI(url) +
                    '" "' +
                    chromiumBrowser +
                    '"', {
                    cwd: _dirname,
                    stdio: 'ignore',
                });
                return true;
            }
            catch (_a) {
                // Ignore errors.
            }
        }
    }
    // Another special case: on OS X, check if BROWSER has been set to "open".
    // In this case, instead of passing `open` to `opn` (which won't work),
    // just ignore it (thus ensuring the intended behavior, i.e. opening the system browser):
    // https://github.com/facebook/create-react-app/pull/1690#issuecomment-283518768
    if (process.platform === 'darwin' && browser === 'open') {
        browser = undefined;
    }
    // Fallback to open
    // (It will always open new tab)
    // eslint-disable-next-line sonar/no-try-promise
    try {
        open(url, {
            app: browser
                ? {
                    name: browser,
                    arguments: args,
                }
                : undefined,
            wait: false,
        }).catch(() => { }); // eslint-disable-line @typescript-eslint/no-empty-function -- Prevent `unhandledRejection` error.
        return true;
    }
    catch (_b) {
        return false;
    }
}
/**
 * Reads the BROWSER environment variable and decides what to do with it. Returns
 * true if it opened a browser or ran a node.js script, otherwise false.
 */
export function openBrowser(url) {
    const { action, value, args } = getBrowserEnv();
    switch (action) {
        case Action.NONE:
            // Special case: BROWSER="none" will prevent opening completely.
            return false;
        case Action.SCRIPT:
            return executeNodeScript(value, url);
        case Action.BROWSER:
            return startBrowserProcess(value, url, args);
        default:
            throw new Error('Not implemented.');
    }
}
//# sourceMappingURL=browser.js.map