/**
 * @license
 * Copyright 2018 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { __decorate } from "tslib";
// Style preference for leading underscores.
// tslint:disable:strip-private-property-underscore
import '@material/mwc-ripple/mwc-ripple';
import { ariaProperty } from '@material/mwc-base/aria-property';
import { addHasRemoveClass, FormElement } from '@material/mwc-base/form-element';
import { observer } from '@material/mwc-base/observer';
import { SingleSelectionController } from '@material/mwc-radio/single-selection-controller';
import { RippleHandlers } from '@material/mwc-ripple/ripple-handlers';
import MDCRadioFoundation from '@material/radio/foundation';
import { html } from 'lit';
import { eventOptions, property, query, queryAsync, state } from 'lit/decorators.js';
import { classMap } from 'lit/directives/class-map.js';
import { ifDefined } from 'lit/directives/if-defined.js';
/**
 * @fires checked
 * @soyCompatible
 */
export class RadioBase extends FormElement {
    constructor() {
        super(...arguments);
        this._checked = false;
        this.useStateLayerCustomProperties = false;
        this.global = false;
        this.disabled = false;
        this.value = 'on';
        this.name = '';
        /**
         * Touch target extends beyond visual boundary of a component by default.
         * Set to `true` to remove touch target added to the component.
         * @see https://material.io/design/usability/accessibility.html
         */
        this.reducedTouchTarget = false;
        this.mdcFoundationClass = MDCRadioFoundation;
        /**
         * input's tabindex is updated based on checked status.
         * Tab navigation will be removed from unchecked radios.
         */
        this.formElementTabIndex = 0;
        this.focused = false;
        this.shouldRenderRipple = false;
        this.rippleElement = null;
        this.rippleHandlers = new RippleHandlers(() => {
            this.shouldRenderRipple = true;
            this.ripple.then((v) => {
                this.rippleElement = v;
            });
            return this.ripple;
        });
    }
    get checked() {
        return this._checked;
    }
    /**
     * We define our own getter/setter for `checked` because we need to track
     * changes to it synchronously.
     *
     * The order in which the `checked` property is set across radio buttons
     * within the same group is very important. However, we can't rely on
     * UpdatingElement's `updated` callback to observe these changes (which is
     * also what the `@observer` decorator uses), because it batches changes to
     * all properties.
     *
     * Consider:
     *
     *   radio1.disabled = true;
     *   radio2.checked = true;
     *   radio1.checked = true;
     *
     * In this case we'd first see all changes for radio1, and then for radio2,
     * and we couldn't tell that radio1 was the most recently checked.
     */
    set checked(isChecked) {
        var _a, _b;
        const oldValue = this._checked;
        if (isChecked === oldValue) {
            return;
        }
        this._checked = isChecked;
        if (this.formElement) {
            this.formElement.checked = isChecked;
        }
        (_a = this._selectionController) === null || _a === void 0 ? void 0 : _a.update(this);
        if (isChecked === false) {
            // Remove focus ring when unchecked on other radio programmatically.
            // Blur on input since this determines the focus style.
            (_b = this.formElement) === null || _b === void 0 ? void 0 : _b.blur();
        }
        this.requestUpdate('checked', oldValue);
        // useful when unchecks self and wrapping element needs to synchronize
        // TODO(b/168543810): Remove triggering event on programmatic API call.
        this.dispatchEvent(new Event('checked', { bubbles: true, composed: true }));
    }
    _handleUpdatedValue(newValue) {
        // the observer function can't access protected fields (according to
        // closure compiler) because it's not a method on the class, so we need this
        // wrapper.
        this.formElement.value = newValue;
    }
    /** @soyTemplate */
    renderRipple() {
        return this.shouldRenderRipple ? html `<mwc-ripple unbounded accent
        .internalUseStateLayerCustomProperties="${this.useStateLayerCustomProperties}"
        .disabled="${this.disabled}"></mwc-ripple>` :
            '';
    }
    get isRippleActive() {
        var _a;
        return ((_a = this.rippleElement) === null || _a === void 0 ? void 0 : _a.isActive) || false;
    }
    connectedCallback() {
        super.connectedCallback();
        // Note that we must defer creating the selection controller until the
        // element has connected, because selection controllers are keyed by the
        // radio's shadow root. For example, if we're stamping in a lit map
        // or repeat, then we'll be constructed before we're added to a root node.
        //
        // Also note if we aren't using native shadow DOM, we still need a
        // SelectionController, because we should update checked status of other
        // radios in the group when selection changes. It also simplifies
        // implementation and testing to use one in all cases.
        //
        // eslint-disable-next-line @typescript-eslint/no-use-before-define
        this._selectionController = SingleSelectionController.getController(this);
        this._selectionController.register(this);
        // Radios maybe checked before connected, update selection as soon it is
        // connected to DOM. Last checked radio button in the DOM will be selected.
        //
        // NOTE: If we update selection only after firstUpdate() we might mistakenly
        // update checked status before other radios are rendered.
        this._selectionController.update(this);
    }
    disconnectedCallback() {
        // The controller is initialized in connectedCallback, so if we are in
        // disconnectedCallback then it must be initialized.
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        this._selectionController.unregister(this);
        this._selectionController = undefined;
    }
    focus() {
        this.formElement.focus();
    }
    createAdapter() {
        return Object.assign(Object.assign({}, addHasRemoveClass(this.mdcRoot)), { setNativeControlDisabled: (disabled) => {
                this.formElement.disabled = disabled;
            } });
    }
    handleFocus() {
        this.focused = true;
        this.handleRippleFocus();
    }
    handleClick() {
        // Firefox has weird behavior with radios if they are not focused
        this.formElement.focus();
    }
    handleBlur() {
        this.focused = false;
        this.formElement.blur();
        this.rippleHandlers.endFocus();
    }
    setFormData(formData) {
        if (this.name && this.checked) {
            formData.append(this.name, this.value);
        }
    }
    /**
     * @soyTemplate
     * @soyAttributes radioAttributes: input
     * @soyClasses radioClasses: .mdc-radio
     */
    render() {
        /** @classMap */
        const classes = {
            'mdc-radio--touch': !this.reducedTouchTarget,
            'mdc-ripple-upgraded--background-focused': this.focused,
            'mdc-radio--disabled': this.disabled,
        };
        return html `
      <div class="mdc-radio ${classMap(classes)}">
        <input
          tabindex="${this.formElementTabIndex}"
          class="mdc-radio__native-control"
          type="radio"
          name="${this.name}"
          aria-label="${ifDefined(this.ariaLabel)}"
          aria-labelledby="${ifDefined(this.ariaLabelledBy)}"
          .checked="${this.checked}"
          .value="${this.value}"
          ?disabled="${this.disabled}"
          @change="${this.changeHandler}"
          @focus="${this.handleFocus}"
          @click="${this.handleClick}"
          @blur="${this.handleBlur}"
          @mousedown="${this.handleRippleMouseDown}"
          @mouseenter="${this.handleRippleMouseEnter}"
          @mouseleave="${this.handleRippleMouseLeave}"
          @touchstart="${this.handleRippleTouchStart}"
          @touchend="${this.handleRippleDeactivate}"
          @touchcancel="${this.handleRippleDeactivate}">
        <div class="mdc-radio__background">
          <div class="mdc-radio__outer-circle"></div>
          <div class="mdc-radio__inner-circle"></div>
        </div>
        ${this.renderRipple()}
      </div>`;
    }
    handleRippleMouseDown(event) {
        const onUp = () => {
            window.removeEventListener('mouseup', onUp);
            this.handleRippleDeactivate();
        };
        window.addEventListener('mouseup', onUp);
        this.rippleHandlers.startPress(event);
    }
    handleRippleTouchStart(event) {
        this.rippleHandlers.startPress(event);
    }
    handleRippleDeactivate() {
        this.rippleHandlers.endPress();
    }
    handleRippleMouseEnter() {
        this.rippleHandlers.startHover();
    }
    handleRippleMouseLeave() {
        this.rippleHandlers.endHover();
    }
    handleRippleFocus() {
        this.rippleHandlers.startFocus();
    }
    changeHandler() {
        this.checked = this.formElement.checked;
    }
}
__decorate([
    query('.mdc-radio')
], RadioBase.prototype, "mdcRoot", void 0);
__decorate([
    query('input')
], RadioBase.prototype, "formElement", void 0);
__decorate([
    state()
], RadioBase.prototype, "useStateLayerCustomProperties", void 0);
__decorate([
    property({ type: Boolean })
], RadioBase.prototype, "global", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], RadioBase.prototype, "checked", null);
__decorate([
    property({ type: Boolean }),
    observer(function (disabled) {
        this.mdcFoundation.setDisabled(disabled);
    })
], RadioBase.prototype, "disabled", void 0);
__decorate([
    property({ type: String }),
    observer(function (value) {
        this._handleUpdatedValue(value);
    })
], RadioBase.prototype, "value", void 0);
__decorate([
    property({ type: String })
], RadioBase.prototype, "name", void 0);
__decorate([
    property({ type: Boolean })
], RadioBase.prototype, "reducedTouchTarget", void 0);
__decorate([
    property({ type: Number })
], RadioBase.prototype, "formElementTabIndex", void 0);
__decorate([
    state()
], RadioBase.prototype, "focused", void 0);
__decorate([
    state()
], RadioBase.prototype, "shouldRenderRipple", void 0);
__decorate([
    queryAsync('mwc-ripple')
], RadioBase.prototype, "ripple", void 0);
__decorate([
    ariaProperty,
    property({ attribute: 'aria-label' })
], RadioBase.prototype, "ariaLabel", void 0);
__decorate([
    ariaProperty,
    property({ attribute: 'aria-labelledby' })
], RadioBase.prototype, "ariaLabelledBy", void 0);
__decorate([
    eventOptions({ passive: true })
], RadioBase.prototype, "handleRippleTouchStart", null);
//# sourceMappingURL=mwc-radio-base.js.map